<?php
session_start();
include '../../includes/db_connection.php'; 

// ব্যবহারকারী লগইন করা আছে কিনা তা নিশ্চিত করা
if (!isset($_SESSION['user_id'])) {
    header("Location: /school_management/public/index.php");
    exit();
}

$student_data = null;
$error = '';

// **ডাটাবেস সংযোগ পরীক্ষা**
if (!isset($conn) || $conn->connect_error) {
    $error = "ডাটাবেস সংযোগে গুরুতর সমস্যা: " . (isset($conn) ? $conn->connect_error : "সংযোগ ভ্যারিয়েবল সেট করা নেই।");
} else {
    // GET রিকোয়েস্ট থেকে স্টুডেন্ট আইডি নেওয়া 
    if (isset($_GET['id']) && is_numeric($_GET['id'])) {
        $student_id = intval($_GET['id']);
        
        // ডাটাবেস থেকে শিক্ষার্থীর সমস্ত তথ্য এবং শ্রেণির নাম সহ কোয়েরি 
        $sql = "SELECT 
                    s.*, c.class_name 
                FROM students s
                JOIN classes c ON s.class_id = c.class_id
                WHERE s.student_id = ?";

        $stmt = $conn->prepare($sql);

        if ($stmt === false) {
            $error = "ডাটাবেস প্রস্তুতি ব্যর্থ: " . $conn->error;
        } else {
            $stmt->bind_param("i", $student_id);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result === false) {
                 $error = "কোয়েরি এক্সিকিউশনে সমস্যা: " . $stmt->error;
            } elseif ($result->num_rows === 1) {
                $student_data = $result->fetch_assoc();
            } else {
                $error = "এই আইডি সহ কোনো শিক্ষার্থী পাওয়া যায়নি। (আইডি: " . $student_id . ")";
            }
            $stmt->close();
        }
    } else {
        $error = "কোনো শিক্ষার্থীর আইডি নির্দিষ্ট করা হয়নি।";
    }
}


// সংযোগ বন্ধ করা
if (isset($conn) && $conn->ping()) {
    $conn->close();
}

// তারিখ ফরম্যাটিং ফাংশন
function formatDate($date) {
    if ($date && $date !== '0000-00-00' && strtotime($date) !== false) {
        return date('d-m-Y', strtotime($date));
    }
    return 'প্রযোজ্য নয় / অজানা';
}
?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>শিক্ষার্থী ভর্তি ফরমের বিস্তারিত ভিউ</title>
    <link rel="stylesheet" href="/school_management/public/css/style.css"> 
    <style>
        /* ফন্ট ও বেসিক সেটআপ */
        body { font-family: 'Arial', sans-serif; background-color: #f0f2f5; }
        
        /* ফরম কন্টেইনার স্টাইল (A4 পেপার ইফেক্ট) */
        .details-container {
            background: #ffffff;
            padding: 40px;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            max-width: 900px; 
            margin: 30px auto;
            border: 2px solid #3498db; 
        }

        /* হেডার ডিজাইন */
        .form-header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 4px double #2c3e50; 
            padding-bottom: 15px;
        }
        .form-header h2 {
            color: #2c3e50;
            margin: 0 0 5px 0;
            font-size: 2.4em; 
            font-weight: 800; 
        }
        .form-header p {
            color: #7f8c8d;
            margin: 0;
            font-size: 1.2em; 
        }
        .form-header .id-box {
            display: inline-block;
            background-color: #e74c3c; 
            color: white;
            padding: 5px 15px;
            border-radius: 5px;
            font-size: 1.2em; 
            font-weight: bold;
            margin-top: 10px;
        }

        /* সেকশন শিরোনাম */
        .details-section-title {
            grid-column: 1 / -1; 
            background-color: #ecf0f1;
            color: #2c3e50;
            padding: 14px 18px; 
            margin-top: 25px;
            margin-bottom: 15px;
            border-left: 6px solid #3498db; 
            font-weight: bold;
            font-size: 1.4em; 
            border-radius: 4px;
        }

        /* গ্রিড বিন্যাস (দুই কলাম) */
        .details-grid {
            display: grid;
            grid-template-columns: 1fr 1fr; 
            gap: 25px 35px; 
            font-size: 16px; 
        }
        
        /* প্রতিটি ফিল্ডের রো */
        .field-row {
            display: flex;
            flex-direction: column;
        }

        /* লেবেল স্টাইল */
        .details-label {
            font-weight: 800; 
            color: #34495e;
            margin-bottom: 5px;
            font-size: 1em; 
            text-transform: uppercase;
        }
        
        /* ডাটা ভ্যালু স্টাইল */
        .details-value {
            color: #2c3e50;
            padding: 12px 15px; 
            border: 1px solid #c9d2d8;
            background-color: #fafafa;
            border-radius: 5px;
            font-size: 1.1em; 
            font-weight: 700; 
            min-height: 45px;
            display: flex;
            align-items: center;
        }
        .details-value strong {
            color: #2980b9;
            font-weight: 900; 
        }
        .details-value.highlight {
            background-color: #e6f7ff; 
            border-color: #91d5ff;
        }

        /* সিঙ্গেল কলামের জন্য ফুল উইডথ */
        .full-width {
            grid-column: 1 / -1;
        }
        
        /* বাটন স্টাইল */
        .action-buttons {
            text-align: center;
            margin-top: 40px;
        }
        .action-buttons a, .action-buttons button {
            display: inline-block;
            width: 220px;
            margin: 10px 5px;
            padding: 14px 25px; 
            border-radius: 5px;
            text-decoration: none;
            font-weight: bold;
            font-size: 16px; 
            transition: background-color 0.3s;
            border: none;
            color: white;
        }
        
        /* নির্দেশনার জন্য নতুন স্টাইল (দ্বিতীয় পৃষ্ঠা) */
        .instruction-page {
            display: none; /* বাই ডিফল্ট ব্রাউজারে এটি লুকানো থাকবে */
            padding: 30px 40px;
            background-color: #ffffff;
            border: 2px solid #2ecc71; 
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            border-radius: 10px;
            max-width: 900px;
            margin: 30px auto;
        }
        .instruction-page h3 {
            text-align: center;
            color: #27ae60;
            font-size: 1.8em;
            border-bottom: 2px solid #2ecc71;
            padding-bottom: 10px;
            margin-bottom: 20px;
        }
        .instruction-page ol {
            list-style-type: decimal;
            padding-left: 25px;
        }
        .instruction-page ol li {
            margin-bottom: 15px;
            line-height: 1.6;
            font-size: 1.1em;
            font-weight: 500;
            color: #34495e;
        }
        .signature-section {
            display: flex;
            justify-content: space-between;
            margin-top: 50px;
            font-weight: bold;
            text-align: center;
        }
        .signature-section div {
            width: 30%;
            border-top: 1px dashed #000;
            padding-top: 10px;
            font-size: 0.9em; /* ওয়েব ভিউতে ছোট রাখা হলো */
        }


        /* ================================================= */
        /* === প্রিন্ট স্টাইল (Print Styles) - এক পেজে ফিট করা === */
        /* ================================================= */
        @media print {
            /* সমস্ত অপ্রয়োজনীয় UI উপাদান লুকান */
            .header, .nav, .footer, .action-buttons { 
                display: none !important; 
            }
            
            /* পেজ সেটিংস */
            @page { 
                size: A4 portrait; /* এ৪ সাইজ পোর্ট্রেট */
                margin: 8mm; /* মার্জিন আরও কমানো হয়েছে */
            }

            /* প্রথম পৃষ্ঠা: মূল ডেটা কন্টেইনার */
            .details-container { 
                box-shadow: none; 
                max-width: 100%; 
                margin: 0; 
                padding: 5mm 8mm 5mm 8mm; /* প্যাডিং কমানো হয়েছে */
                border: 1px solid #000 !important; /* প্রিন্ট বর্ডার */
                -webkit-print-color-adjust: exact; 
                color-adjust: exact;
                page-break-after: always; /* **ফর্ম শেষ হওয়ার পর নতুন পৃষ্ঠা শুরু হবে** */
            }
            
            /* ছবির অবস্থান ঠিক করা - মূল সমস্যাটি সম্ভবত এখানে ছিল। মার্জিন কমানো হলো। */
            div[style*="margin-bottom: -150px"] {
                margin-bottom: -110px !important; /* ছবির জন্য মার্জিন কমানো হয়েছে */
                margin-right: 15px !important;
            }
            img[alt="শিক্ষার্থীর ছবি"] {
                width: 100px !important; /* ছবি ছোট করা হয়েছে */
                height: 100px !important;
                border: 2px solid #3498db !important;
            }

            /* গ্রিড লেআউট - ফন্ট ও গ্যাপ কমানো */
            .details-grid { 
                grid-template-columns: repeat(2, 1fr); 
                gap: 10px 20px; 
                font-size: 12px; 
            }
            .details-label { 
                font-size: 11px; 
                margin-bottom: 1px; 
            }
            .details-value { 
                padding: 5px 8px; 
                border: 1px solid #000;
                background-color: #fff !important; 
                font-size: 12px; 
                font-weight: 600; 
                min-height: unset;
            }
            .details-section-title {
                margin-top: 10px;
                margin-bottom: 5px;
                padding: 6px 8px;
                font-size: 1.1em;
                background-color: #f0f0f0 !important;
                -webkit-print-color-adjust: exact;
                color-adjust: exact;
            }
            
            /* দ্বিতীয় পৃষ্ঠা: নির্দেশনা */
            .instruction-page {
                display: block !important;
                page-break-before: always; /* **নিশ্চিতভাবে নতুন পৃষ্ঠায় শুরু হবে** */
                box-shadow: none; 
                border: 1px solid #000 !important; /* ফর্মের চেয়ে হালকা বর্ডার */
                max-width: 100%; 
                margin: 0;
                padding: 5mm 8mm 5mm 8mm; 
                background-color: #ffffff !important;
                -webkit-print-color-adjust: exact; 
                color-adjust: exact;
            }
            .instruction-page h3 {
                color: #000 !important;
                font-size: 1.4em;
                border-bottom: 1px solid #000 !important;
                padding-bottom: 5px;
                margin-bottom: 10px;
            }
            .instruction-page ol li {
                font-size: 12px;
                color: #000;
                margin-bottom: 8px;
            }
            .signature-section {
                margin-top: 20px;
                font-size: 11px;
            }
            .signature-section div {
                padding-top: 5px;
            }
        }
    </style>
</head>
<body>
<div class="header">
    <div class="nav">
        <a href="/school_management/public/dashboard.php">ড্যাশবোর্ড</a>
        <a href="students_list.php">শিক্ষার্থী তালিকা</a>
        <a href="/school_management/public/logout.php">লগআউট</a>
    </div>
    <h1>শিক্ষার্থী বিস্তারিত ভিউ</h1>
</div>
<div class="content">
    
    <?php if ($error): ?>
        <p style='color:red; background-color: #f8d7da; padding: 10px; border-radius: 4px; max-width: 800px; margin: 20px auto; font-weight: bold;'>
            🛑 ত্রুটি: <?php echo htmlspecialchars($error); ?>
            <br>
            যদি ভর্তির তথ্য না দেখা যায়, তবে নিশ্চিত করুন আপনার `students` টেবিলে `enrollment_date` এবং `fixed_fee` কলাম দুটি আছে।
        </p>
        <div class="action-buttons">
            <a href="students_list.php" class="action-button" style="background-color: #f39c12;">শিক্ষার্থী তালিকায় ফিরে যান</a>
        </div>
    <?php elseif ($student_data): ?>
        
        <div class="details-container">
            <div class="form-header">
                <h2>মিফতাহুন নাজাত মহিলা মাদরাসা</h2>
                <p>শিক্ষার্থী ভর্তি ফরম - বিস্তারিত তথ্য</p>
                <div class="id-box">আইডি নং: <?php echo htmlspecialchars($student_data['student_id']); ?></div>
            </div>

            <?php 
            $photo_path = $student_data['photo_path'] ?? '';
            if (!empty($photo_path)): 
                $photo_url = "/school_management/public/" . ltrim($photo_path, '/');
            ?>
                <div style="text-align: right; margin-bottom: -150px; margin-right: 20px; position: relative; z-index: 1;">
                    <img src="<?php echo htmlspecialchars($photo_url); ?>" 
                         alt="শিক্ষার্থীর ছবি" 
                         style="width: 130px; height: 130px; object-fit: cover; border: 4px solid #3498db; padding: 2px; background: #fff; border-radius: 5px;">
                </div>
            <?php endif; ?>


            <div class="details-grid">
                
                <div class="details-section-title">১. প্রাথমিক ও ভর্তি তথ্য</div>
                
                <div class="field-row">
                    <div class="details-label">শ্রেণি</div>
                    <div class="details-value highlight"><strong><?php echo htmlspecialchars($student_data['class_name'] ?? 'N/A'); ?></strong></div>
                </div>
                
                <div class="field-row">
                    <div class="details-label">শ্রেণি রোল</div>
                    <div class="details-value highlight"><strong><?php echo htmlspecialchars($student_data['class_roll'] ?? 'N/A'); ?></strong></div>
                </div>

                <div class="field-row">
                    <div class="details-label">ভর্তির তারিখ</div>
                    <div class="details-value"><?php echo formatDate($student_data['enrollment_date'] ?? null); ?></div>
                </div>

                <div class="field-row">
                    <div class="details-label">মাসিক বেতন (ধার্যকৃত)</div>
                    <div class="details-value">৳ <?php echo htmlspecialchars(number_format($student_data['fixed_fee'] ?? 0, 0)); ?></div>
                </div>
                
                <div class="details-section-title">২. শিক্ষার্থীর ব্যক্তিগত বিবরণ</div>

                <div class="field-row">
                    <div class="details-label">পূর্ণ নাম</div>
                    <div class="details-value"><?php echo htmlspecialchars(($student_data['first_name'] ?? '') . ' ' . ($student_data['last_name'] ?? '')); ?></div>
                </div>
                
                <div class="field-row">
                    <div class="details-label">জন্ম তারিখ</div>
                    <div class="details-value"><?php echo formatDate($student_data['date_of_birth'] ?? null); ?></div>
                </div>
                
                <div class="field-row">
                    <div class="details-label">লিঙ্গ</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['gender'] ?? 'N/A'); ?></div>
                </div>
                
                <div class="field-row">
                    <div class="details-label">জন্ম নিবন্ধন নং</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['birth_reg_no'] ?? 'N/A'); ?></div>
                </div>

                <div class="field-row full-width">
                    <div class="details-label">পূর্ববর্তী প্রতিষ্ঠানের নাম</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['previous_institute'] ?? 'নেই'); ?></div>
                </div>
                
                <div class="field-row full-width">
                    <div class="details-label">শিক্ষার্থীর মোবাইল নং</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['phone_number'] ?? 'N/A'); ?></div>
                </div>

                <div class="details-section-title">৩. অভিভাবকের তথ্য</div>
                
                <div class="field-row">
                    <div class="details-label">পিতার নাম</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['father_name'] ?? 'N/A'); ?></div>
                </div>
                
                <div class="field-row">
                    <div class="details-label">মাতার নাম</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['mother_name'] ?? 'N/A'); ?></div>
                </div>
                
                <div class="field-row full-width">
                    <div class="details-label">অভিভাবকের মোবাইল নং (যদি থাকে)</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['guardian_phone'] ?? $student_data['phone_number'] ?? 'N/A'); ?></div>
                </div>

                <div class="details-section-title">৪. বর্তমান ঠিকানা</div>
                
                <div class="field-row">
                    <div class="details-label">গ্রাম/মহল্লা</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['village'] ?? 'N/A'); ?></div>
                </div>
                
                <div class="field-row">
                    <div class="details-label">উপজেলা/থানা</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['upazila'] ?? 'N/A'); ?></div>
                </div>
                
                <div class="field-row">
                    <div class="details-label">জেলা</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['district'] ?? 'N/A'); ?></div>
                </div>

                <div class="details-section-title">৫. অতিরিক্ত মন্তব্য</div>
                <div class="field-row full-width">
                    <div class="details-label">অন্যান্য/বিশেষ মন্তব্য (Address কলাম থেকে দেখানো হলো)</div>
                    <div class="details-value"><?php echo htmlspecialchars($student_data['address'] ?? $student_data['remarks'] ?? 'কোনো মন্তব্য নেই'); ?></div>
                </div>
                
            </div>
            
            <div class="action-buttons">
                <a href="students_list.php" class="action-button" style="background-color: #2c3e50;">শিক্ষার্থী তালিকায় ফিরে যান</a>
                <a href="edit_student.php?id=<?php echo htmlspecialchars($student_data['student_id']); ?>" class="action-button" style="background-color: #007bff;">তথ্য সম্পাদনা করুন</a>
                <button onclick="window.print()" class="action-button" style="background-color: #27ae60;">ভর্তি ফরম প্রিন্ট করুন 🖨️</button>
            </div>
        </div>
        
    <?php endif; ?>
</div>

<?php if ($student_data): ?>
<div class="instruction-page">
    <h3>ভর্তি সংক্রান্ত গুরুত্বপূর্ণ নির্দেশনাবলী ও অঙ্গীকার</h3>

    <ol>
        <li>শিক্ষার্থীকে মাদরাসার সকল নিয়ম-কানুন ও শৃঙ্খলা মেনে চলতে হবে। কোনো রকম শৃঙ্খলা ভঙ্গ সহ্য করা হবে না।</li>
        <li>মাসিক বেতন প্রতি মাসের ১০ তারিখের মধ্যে পরিশোধ করতে হবে। অন্যথায় জরিমানা ধার্য করা হবে।</li>
        <li>প্রতিষ্ঠান কর্তৃপক্ষ কর্তৃক আয়োজিত সকল ধর্মীয় ও শিক্ষামূলক কার্যক্রমে অংশগ্রহণ করা বাধ্যতামূলক।</li>
        <li>শিক্ষার্থীর সকল প্রকার অনুপস্থিতি অভিভাবককে লিখিত বা ফোন কলের মাধ্যমে জানাতে হবে। বিনা নোটিশে দীর্ঘ অনুপস্থিতি গ্রহণযোগ্য নয়।</li>
        <li>পরীক্ষার ফি ও অন্যান্য আবশ্যকীয় ফি নির্দিষ্ট সময়ের মধ্যে পরিশোধ করতে হবে।</li>
        <li>প্রতিষ্ঠানের পোশাক পরিধান করে মাদরাসায় আসতে হবে এবং পোশাকের পরিচ্ছন্নতা বজায় রাখতে হবে।</li>
        <li>শিক্ষার্থী মাদরাসার কোনো সম্পত্তির ক্ষতিসাধন করলে, অভিভাবক ক্ষতিপূরণ দিতে বাধ্য থাকবেন।</li>
        <li>প্রতিষ্ঠানের প্রয়োজনে কর্তৃপক্ষ যেকোনো সময় নিয়মের পরিবর্তন, সংশোধন ও পরিবর্ধন করার ক্ষমতা সংরক্ষণ করে।</li>
    </ol>
    
    <div style="text-align: justify; margin-top: 30px; font-style: italic; font-size: 1.1em; color: #555; font-weight: 500;">
        আমি (অভিভাবক) এই মর্মে অঙ্গীকার করছি যে, আমার সন্তান/ওয়ার্ড মিফতাহুন নাজাত মহিলা মাদরাসার সকল নিয়ম-কানুন ও শৃঙ্খলা মেনে চলবে এবং আমি নিয়মিত তার খোঁজ-খবর রাখব।
    </div>

    <div class="signature-section">
        <div>
            শিক্ষার্থীর স্বাক্ষর
        </div>
        <div>
            অভিভাবকের স্বাক্ষর ও তারিখ
        </div>
        <div>
            কর্তৃপক্ষের স্বাক্ষর
        </div>
    </div>
</div>
<?php endif; ?>

<div class="footer">
    <p>&copy; <?php echo date("Y"); ?> স্কুল ম্যানেজমেন্ট সিস্টেম। সর্বস্বত্ব সংরক্ষিত।</p>
</div>
</body>
</html>