<?php
session_start();
// ডাটাবেস সংযোগ ফাইল যুক্ত করা
include '../../includes/db_connection.php';
date_default_timezone_set('Asia/Dhaka');

// নিশ্চিত করুন যে ইউজার লগইন করা আছে এবং তার ভূমিকা (Role) ঠিক আছে
if (!isset($_SESSION['user_id']) || ($_SESSION['role'] != 'Admin' && $_SESSION['role'] != 'Teacher')) {
    header("Location: ../../public/index.php"); // লগইন পেজে রিডাইরেক্ট
    exit();
}

// ==========================================
// ১. শিক্ষার্থীর আইডি গ্রহণ এবং যাচাই
// ==========================================

$student_id = isset($_GET['student_id']) ? intval($_GET['student_id']) : 0;
$student_data = null;
$error_message = '';
$issue_date = date('d F Y', strtotime('now')); // ইস্যুর তারিখ

// বাংলা মাস ও সংখ্যা ম্যাপিং
$bangla_months = array(
    'January' => 'জানুয়ারি', 'February' => 'ফেব্রুয়ারি', 'March' => 'মার্চ',
    'April' => 'এপ্রিল', 'May' => 'মে', 'June' => 'জুন',
    'July' => 'জুলাই', 'August' => 'আগস্ট', 'September' => 'সেপ্টেম্বর',
    'October' => 'অক্টোবর', 'November' => 'নভেম্বর', 'December' => 'ডিসেম্বর'
);
$bangla_numbers = array('১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯', '০');
$english_numbers = array('1', '2', '3', '4', '5', '6', '7', '8', '9', '0');

// ইংরেজি থেকে বাংলায় সংখ্যা রূপান্তরের ফাংশন
function convert_to_bangla_number($number) {
    global $bangla_numbers, $english_numbers;
    return str_replace($english_numbers, $bangla_numbers, $number);
}

// ইংরেজি তারিখকে বাংলায় রূপান্তরের জন্য প্রস্তুত করা
$issue_date_bn = str_replace(array_keys($bangla_months), array_values($bangla_months), $issue_date);
$issue_date_bn = convert_to_bangla_number($issue_date_bn);


if ($student_id > 0) {
    // ✅ ডাটাবেস থেকে শিক্ষার্থীর তথ্য আনা (first_name, last_name ব্যবহার করা হয়েছে)
    $sql = "SELECT 
                s.student_id, 
                s.first_name, 
                s.last_name, 
                s.father_name, 
                s.mother_name, 
                s.birth_reg_no, /* birth_reg_no কে registration_number হিসেবে ব্যবহার করা হলো */
                s.enrollment_date, /* enrollment_date কে admission_date হিসেবে ব্যবহার করা হলো */
                c.class_name
            FROM students s
            LEFT JOIN classes c ON s.class_id = c.class_id
            WHERE s.student_id = ?";
    
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $student_id);
        
        // ক্যোয়ারি সফলভাবে প্রস্তুত হলে এটি সম্পাদন করা হবে
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            
            if ($result->num_rows === 1) {
                $student_data = $result->fetch_assoc();
                
                // ✅ first_name এবং last_name একত্রিত করে পূর্ণাঙ্গ নাম তৈরি
                $full_name = htmlspecialchars($student_data['first_name'] . ' ' . $student_data['last_name']);
                
                // ভর্তির তারিখ (enrollment_date) বাংলায় রূপান্তর
                $enrollment_date_bn = str_replace(array_keys($bangla_months), array_values($bangla_months), date('d F Y', strtotime($student_data['enrollment_date'])));
                $enrollment_date_bn = convert_to_bangla_number($enrollment_date_bn);
                
                // জন্ম নিবন্ধন নম্বর (birth_reg_no) বাংলায় রূপান্তর
                $birth_reg_no_bn = convert_to_bangla_number($student_data['birth_reg_no']);

                
                // ডেটা টেমপ্লেটের জন্য প্রস্তুত
                $student_data['name_bn'] = $full_name;
                $student_data['father_name_bn'] = htmlspecialchars($student_data['father_name']);
                $student_data['mother_name_bn'] = htmlspecialchars($student_data['mother_name']);
                $student_data['class_name_bn'] = htmlspecialchars($student_data['class_name']);
                $student_data['enrollment_date_bn'] = $enrollment_date_bn;
                $student_data['birth_reg_no_bn'] = $birth_reg_no_bn;
                
            } else {
                $error_message = "এই আইডি'র কোনো শিক্ষার্থী পাওয়া যায়নি। (ID: " . $student_id . ")";
            }
        } else {
            // ক্যোয়ারি এক্সিকিউট করতে ব্যর্থ হলে
            $error_message = "ডাটাবেস ক্যোয়ারি এক্সিকিউট করতে ব্যর্থ: " . $stmt->error;
        }
        $stmt->close();
    } else {
        // ক্যোয়ারি প্রস্তুত করতে ব্যর্থ হলে
        $error_message = "ডাটাবেস ক্যোয়ারি প্রস্তুত করতে ব্যর্থ: " . $conn->error;
    }
} else {
    $error_message = "প্রত্যয়নপত্র তৈরি করতে একটি সঠিক শিক্ষার্থীর আইডি প্রয়োজন।";
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>প্রত্যয়নপত্র | <?php echo $student_data ? $student_data['name_bn'] : 'ত্রুটি'; ?></title>
    <style>
        body {
            font-family: 'Arial', sans-serif;
            background-color: #f4f4f4;
            display: flex;
            flex-direction: column;
            align-items: center;
            padding: 20px;
        }
        .container {
            max-width: 900px;
            width: 100%;
            background: white;
            padding: 40px;
            border: 1px solid #ddd;
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.1);
            position: relative;
        }
        /* প্রত্যয়নপত্রের ডিজাইন */
        .testimonial-body {
            border: 5px double #34495e; /* চার্টারের বর্ডার */
            padding: 50px;
            min-height: 800px; /* প্রিন্ট করার জন্য পর্যাপ্ত উচ্চতা */
            position: relative;
        }
        .header {
            text-align: center;
            margin-bottom: 40px;
            border-bottom: 1px solid #ccc;
            padding-bottom: 20px;
        }
        .header h1 {
            font-size: 2.2em;
            color: #2c3e50;
            margin-bottom: 5px;
        }
        .header h3 {
            font-size: 1.5em;
            color: #3498db;
            margin-top: 0;
            text-transform: uppercase;
        }
        .cert-title {
            text-align: center;
            font-size: 2em;
            font-weight: bold;
            color: #e67e22;
            margin: 30px 0;
            text-decoration: underline;
        }
        .content {
            font-size: 1.2em;
            text-align: justify;
            line-height: 2.0;
        }
        .content strong {
            color: #34495e;
            font-weight: bold;
        }
        .signature-section {
            display: flex;
            justify-content: space-between;
            margin-top: 100px;
            font-size: 1.1em;
        }
        .signature-item {
            text-align: center;
            width: 30%;
            border-top: 1px solid #333;
            padding-top: 10px;
        }
        .date-location {
            position: absolute;
            top: 250px;
            left: 50px;
            font-size: 1.1em;
            font-weight: bold;
            color: #7f8c8d;
        }
        
        /* প্রিন্ট এবং ব্যাক বাটন স্টাইল */
        .action-buttons {
            margin-bottom: 20px;
            text-align: right;
            width: 900px;
            max-width: 100%;
        }
        .action-btn {
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-weight: bold;
            margin-left: 10px;
        }
        .print-btn {
            background-color: #2ecc71;
            color: white;
        }
        .back-btn {
            background-color: #f39c12;
            color: white;
            text-decoration: none;
        }

        /* প্রিন্টের জন্য CSS */
        @media print {
            body { background: white; }
            .action-buttons { display: none; }
            .container {
                box-shadow: none;
                padding: 0;
                border: none;
                margin: 0;
            }
            .testimonial-body {
                border: 5px double #34495e;
                min-height: 100vh;
                padding: 50px;
            }
        }
    </style>
</head>
<body>

<div class="action-buttons">
    <a href="students_list.php" class="action-btn back-btn">
        ← শিক্ষার্থী তালিকায় ফিরে যান
    </a>
    <button onclick="window.print()" class="action-btn print-btn">
        🖨️ প্রিন্ট করুন
    </button>
</div>

<div class="container">
    <?php if ($student_data): ?>
    <div class="testimonial-body">
        <div class="header">
            <h1>মিফতাহুন নাজাত মহিলা মাদরাসা</h1>
            <h3>প্রত্যয়নপত্র</h3>
            <p style="font-size: 0.9em; color: #7f8c8d;">জন্ম নিবন্ধন নং: <?php echo $student_data['birth_reg_no_bn']; ?></p>
        </div>

        <div class="date-location">
            তারিখ: <?php echo $issue_date_bn; ?>
        </div>

        <div class="cert-title">
            এই মর্মে প্রত্যয়ন করা হচ্ছে
        </div>

        <div class="content">
            <p>
                যে, **<?php echo $student_data['name_bn']; ?>** আমাদের মাদরাসার একজন নিয়মিত শিক্ষার্থী। তিনি **<?php echo $student_data['father_name_bn']; ?>** এবং **<?php echo $student_data['mother_name_bn']; ?>** এর সুযোগ্য সন্তান।
            </p>
            <p>
                তার জন্ম নিবন্ধন নম্বর হলো **<?php echo $student_data['birth_reg_no_bn']; ?>**। সে বিগত **<?php echo $student_data['enrollment_date_bn']; ?>** তারিখে আমাদের মাদরাসায় ভর্তি হয় এবং **<?php echo $student_data['class_name_bn']; ?>** শ্রেণিতে সুনামের সাথে অধ্যয়ন করেছে।
            </p>
            <p>
                আমার জানামতে, মাদরাসায় অধ্যয়নকালীন সময়ে সে কোনো প্রকার রাষ্ট্রবিরোধী বা শৃঙ্খলা পরিপন্থী কার্যকলাপে জড়িত ছিল না। তার চারিত্রিক বৈশিষ্ট্য সর্বাবস্থায় উত্তম ছিল।
            </p>
            <p>
                আমি তার উজ্জ্বল ভবিষ্যৎ কামনা করি।
            </p>
        </div>

        <div class="signature-section">
            <div class="signature-item">
                <p>শিক্ষার্থীর স্বাক্ষর</p>
            </div>
            <div class="signature-item">
                <p>অফিস সহকারী</p>
            </div>
            <div class="signature-item">
                <p style="margin-bottom: 30px;">অধ্যক্ষের স্বাক্ষর ও সীল</p>
                <p style="margin-top: 0; font-weight: bold; color: #34495e;">আলহাজ্ব মাওলানা (অধ্যক্ষের নাম)</p>
            </div>
        </div>
        
    </div>
    <?php else: ?>
    <div style="text-align: center; color: red; padding: 50px; border: 1px solid red; background-color: #ffe8e8;">
        <h2>ত্রুটি!</h2>
        <p>ডাটাবেস বা ক্যোয়ারি সমস্যা। অনুগ্রহ করে নিচের বার্তাটি দেখুন:</p>
        <p style="font-size: 0.9em; font-weight: bold; color: #c0392b;"><?php echo $error_message; ?></p>
        <p>ডাটাবেস সংযোগ বা কলামের নাম ঠিক আছে কিনা যাচাই করুন।</p>
    </div>
    <?php endif; ?>
</div>

</body>
</html>