<?php
session_start();
$ds = DIRECTORY_SEPARATOR;
$project_root_raw = dirname(dirname(__DIR__)); 
$project_root_1 = rtrim($project_root_raw, '/\\'); 

$db_path = $project_root_1 . $ds . 'includes' . $ds . 'db_connection.php';
$functions_path = $project_root_1 . $ds . 'includes' . $ds . 'functions.php';

if (file_exists($db_path) && file_exists($functions_path)) {
    include $db_path;
    include $functions_path;
} else {
    die("FATAL ERROR: Required core files not found.");
}

// *** গ্রেড গণনা ফাংশন (অবশ্যই প্রয়োজন, তাই উপরে রাখা হলো) ***
if (!function_exists('calculate_grade')) {
    function calculate_grade($mark) {
        if ($mark >= 80) return 'A+';
        if ($mark >= 70) return 'A';
        if ($mark >= 60) return 'A-';
        if ($mark >= 50) return 'B';
        if ($mark >= 40) return 'C';
        if ($mark >= 33) return 'D';
        return 'F';
    }
}

/**
 * বিষয়ভিত্তিক পারফরম্যান্স বিশ্লেষণ ফাংশন
 * তিনটি পরীক্ষার নম্বরের গড়ের উপর ভিত্তি করে মন্তব্য করে।
 * @param array $marks_array পরীক্ষার প্রাপ্ত নম্বরের অ্যারে
 * @return string বিশ্লেষণমূলক মন্তব্য
 */
function analyze_performance(array $marks_array) {
    // শুধুমাত্র যেখানে নম্বর আছে সেগুলো ফিল্টার করা
    $valid_marks = array_filter($marks_array, function($m) {
        return is_numeric($m) && $m >= 0;
    });

    if (empty($valid_marks)) {
        return '<span class="analysis-pending">তথ্য নেই</span>';
    }

    $total_valid_exams = count($valid_marks);
    $average_mark = array_sum($valid_marks) / $total_valid_exams;

    if ($average_mark >= 80) {
        return '<span class="analysis-excellent">অসাধারণ, খুব ভালো</span>';
    } elseif ($average_mark >= 65) {
        return '<span class="analysis-good">সন্তোষজনক উন্নতি</span>';
    } elseif ($average_mark >= 50) {
        return '<span class="analysis-average">আরও মনোযোগ প্রয়োজন</span>';
    } else {
        return '<span class="analysis-poor">অবশ্যই উন্নতি প্রয়োজন</span>';
    }
}
// ************************************************************


// লগইন ও অনুমতি চেক
if (!is_logged_in() || (isset($_SESSION['role']) && $_SESSION['role'] !== 'Admin')) {
    redirect_to('../../index.php'); 
}

// ইনপুট ডেটা সংগ্রহ ও বৈধতা যাচাই
$student_id = filter_input(INPUT_GET, 'student_id', FILTER_VALIDATE_INT);
$class_id = filter_input(INPUT_GET, 'class_id', FILTER_VALIDATE_INT); 

$student_data = null;
$exam_results_grid = []; // গ্রিড ডেটা: Subject Name => [Exam ID => [Marks, Grade]]
$unique_exams = []; // সকল পরীক্ষার নাম সংরক্ষণ করার জন্য (ID => Name)
$exam_ids = []; // পরীক্ষার ID-এর ক্রম (sorting এর জন্য)
$error_message = null;

if (!$student_id || !$class_id) {
    $error_message = "অসম্পূর্ণ ডেটা: শিক্ষার্থী বা ক্লাস আইডি অনুপস্থিত।";
}

if (!isset($error_message) && $conn) {
    
    // --- ১. শিক্ষার্থীর তথ্য লোড করা ---
    $sql_student = "
        SELECT 
            s.student_id,
            COALESCE(CONCAT(s.first_name, ' ', s.last_name), 'নাম নেই') AS full_name, 
            s.class_roll,
            c.class_name
        FROM 
            students s
        JOIN 
            classes c ON s.class_id = c.class_id
        WHERE 
            s.student_id = ? AND s.class_id = ?
    ";
    
    // (পস্তুতকরণ ও সম্পাদনের লজিক অপরিবর্তিত)
    if ($stmt = $conn->prepare($sql_student)) {
        $stmt->bind_param("ii", $student_id, $class_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $student_data = $result->fetch_assoc();
        }
        $stmt->close();
    } else {
        $error_message = "শিক্ষার্থী তথ্য ক্যোয়ারি ত্রুটি: " . $conn->error;
        error_log("SQL Error (Student Info): " . $conn->error);
    }
    
    if (!$student_data) {
        $error_message = "শিক্ষার্থীর তথ্য খুঁজে পাওয়া যায়নি। আইডি সঠিক কিনা নিশ্চিত করুন।";
    }

    // --- ২. সকল পরীক্ষার ফলাফল লোড করা ---
    if (!$error_message && $student_data) {
        $sql_all_results = "
            SELECT 
                r.marks, 
                s.subject_name,
                e.exam_name,
                e.exam_id
            FROM 
                exam_results r 
            JOIN 
                subjects s ON r.subject_id = s.subject_id
            JOIN 
                exams e ON r.exam_id = e.exam_id
            WHERE 
                r.student_id = ? 
            ORDER BY 
                e.exam_id ASC, s.subject_name ASC
        ";
        
        if ($stmt = $conn->prepare($sql_all_results)) {
            $stmt->bind_param("i", $student_id); 
            
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                while ($row = $result->fetch_assoc()) {
                    $subject_name = htmlspecialchars($row['subject_name']);
                    $exam_id = (int)$row['exam_id'];
                    $marks = (int)$row['marks'];
                    
                    // পরীক্ষার ডেটা গ্রিডে সাজানো
                    $exam_results_grid[$subject_name][$exam_id] = [
                        'marks' => $marks,
                        'grade' => calculate_grade($marks) 
                    ];
                    
                    // অনন্য পরীক্ষার তালিকা তৈরি
                    $unique_exams[$exam_id] = htmlspecialchars($row['exam_name']);
                    if (!in_array($exam_id, $exam_ids)) {
                        $exam_ids[] = $exam_id;
                    }
                }
            } else {
                $error_message = "ফলাফল ক্যোয়ারি এক্সিকিউশন ত্রুটি: " . $stmt->error;
                error_log("SQL Error (All Results): " . $stmt->error);
            }
            $stmt->close();
        } else {
            $error_message = "ফলাফল ক্যোয়ারি প্রস্তুতি ত্রুটি: " . $conn->error;
            error_log("SQL Error (Prepare All Results): " . $conn->error);
        }
        
        if (empty($exam_results_grid)) {
             $error_message = "এই শিক্ষার্থীর জন্য কোনো পরীক্ষার ফলাফল জমা দেওয়া হয়নি।";
        }
    }
    
    $conn->close();
}

// পরীক্ষার ID-এর ক্রম ঠিক রাখতে সাজানো
sort($exam_ids);

// মোট পরীক্ষার সংখ্যা, টেবিলের কলাম স্প্যান নির্ধারণ করবে
$exam_count = count($exam_ids);

?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>তুলনামূলক মার্কশিট</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        /* সাধারণ ডিজাইন */
        body { background-color: #f0f2f5; font-family: 'Arial', sans-serif; padding: 15px; }
        .container { width: 100%; max-width: 1400px; margin: 0 auto; background-color: #ffffff; border-radius: 15px; box-shadow: 0 8px 20px rgba(0, 0, 0, 0.15); padding: 30px; }
        .header { text-align: center; border-bottom: 4px solid #3498db; padding-bottom: 15px; margin-bottom: 20px; }
        .header h1 { color: #2c3e50; margin: 0; font-size: 2.2em; }
        .header h2 { color: #3498db; margin: 5px 0 10px 0; font-size: 1.5em; }
        
        /* শিক্ষার্থীর তথ্য */
        .student-info { display: flex; justify-content: space-around; flex-wrap: wrap; margin-bottom: 25px; padding: 15px; background-color: #ecf0f1; border-radius: 8px; font-weight: bold; }
        .student-info div { flex: 1 1 200px; margin: 5px 10px; font-size: 1.1em; color: #34495e; }
        .student-info strong { color: #2c3e50; margin-right: 5px; }

        /* টেবিল স্টাইল */
        table { width: 100%; border-collapse: collapse; margin-bottom: 30px; table-layout: fixed; /* কলামের প্রস্থ ফিক্সড করার জন্য */ }
        table th, table td { border: 1px solid #bdc3c7; padding: 10px 5px; text-align: center; font-size: 0.9em; }
        table thead th { background-color: #3498db; color: white; font-weight: bold; }
        table tbody tr:nth-child(even) { background-color: #f7f9fb; }
        
        /* কলামের প্রস্থ নিয়ন্ত্রণ (মোট কলামের সংখ্যা অনুযায়ী সামঞ্জস্য করা হয়েছে) */
        .subject-col { text-align: left; font-weight: bold; width: 15%; /* বিষয় কলামের প্রস্থ */ background-color: #f0f0f0; }
        .marks-col { background-color: #e6ffe6; font-weight: 700; width: 8%; } /* প্রতিটি নম্বর কলাম */
        .grade-col { background-color: #fff0e6; width: 8%; } /* প্রতিটি গ্রেড কলাম */
        .analysis-col { font-weight: bold; width: 20%; /* বিশ্লেষণের জন্য বড় প্রস্থ */ background-color: #ffe6e6; } 

        /* বিশ্লেষণের রং */
        .analysis-excellent { color: #2ecc71; }
        .analysis-good { color: #3498db; }
        .analysis-average { color: #f39c12; }
        .analysis-poor { color: #e74c3c; }
        .analysis-pending { color: #95a5a6; }
        
        /* মন্তব্য এবং স্বাক্ষর বিভাগ */
        .comments-and-signature { display: flex; justify-content: space-between; flex-wrap: wrap; margin-top: 40px; padding-top: 20px; border-top: 1px dashed #bdc3c7; }
        .comment-box { width: 48%; min-height: 120px; margin-bottom: 20px; padding: 15px; border: 1px solid #95a5a6; border-radius: 8px; background-color: #f7f7f7; box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05); }
        .comment-box strong { display: block; margin-bottom: 10px; color: #3498db; border-bottom: 2px solid #3498db; padding-bottom: 5px; }
        .signature-section { width: 100%; display: flex; justify-content: flex-end; margin-top: 40px; padding-top: 20px; }
        .signature-line { display: block; border-top: 2px solid #2c3e50; padding: 5px 0 0; width: 250px; text-align: center; font-weight: bold; color: #2c3e50; }
        
        /* অ্যাকশন বাটন */
        .action-buttons { display: flex; justify-content: flex-start; margin-bottom: 20px; gap: 15px; }
        .btn { padding: 10px 20px; border: none; border-radius: 8px; cursor: pointer; font-size: 1em; font-weight: bold; text-decoration: none; display: inline-flex; align-items: center; gap: 8px; transition: background-color 0.3s; }
        .btn-back { background-color: #e74c3c; color: white; }
        .btn-print { background-color: #2ecc71; color: white; }
        .btn-secondary { background-color: #f39c12; color: white; }
        
        /* প্রিন্ট স্টাইল */
        @media print {
            body { background-color: white !important; padding: 0; margin: 0; }
            .container { box-shadow: none; border: none; padding: 10mm; max-width: 100%; }
            .action-buttons { display: none; }
            table { table-layout: auto; } /* প্রিন্টে কলাম সাইজ নিজে ঠিক করুক */
            table th, table td { border-color: #333 !important; padding: 8px 4px; font-size: 0.8em; }
            table thead th { background-color: #e8e8e8 !important; color: #000 !important; -webkit-print-color-adjust: exact; print-color-adjust: exact; }
            .marks-col, .grade-col, .subject-col, .student-info, .analysis-col { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
            .comment-box { border: 1px solid #333 !important; background-color: #fff !important; box-shadow: none !important; }
            .signature-line { border-top: 1px solid #333 !important; }
        }
        @media (max-width: 600px) {
            .student-info { flex-direction: column; }
            .comment-box { width: 100%; }
            table { table-layout: auto; }
        }
    </style>
</head>
<body>

<div class="container">
    
    <?php if (!isset($error_message)): ?>
        <div class="action-buttons">
            <a href="../../public/dashboard.php" class="btn btn-back" style="background-color: #555;">
                <i class="fas fa-home"></i> হোম
            </a>
            <button onclick="window.print()" class="btn btn-print">
                <i class="fas fa-print"></i> প্রিন্ট রিপোর্ট
            </button>
            <?php 
            $last_exam_id = end($exam_ids) ?? null;
            if ($last_exam_id): 
            ?>
            <a href="generate_mark_sheet.php?student_id=<?php echo $student_id; ?>&exam_id=<?php echo $last_exam_id; ?>&class_id=<?php echo $class_id; ?>" class="btn btn-secondary">
                <i class="fas fa-file-alt"></i> শেষ পরীক্ষার মার্কশিট
            </a>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <div class="mark-sheet">
    <?php if (isset($error_message)): ?>
        <div style="text-align: center; padding: 50px;">
            <div style="background-color: #fce4e4; color: #c0392b; border: 1px solid #c0392b; padding: 20px; border-radius: 8px; font-size: 1.2em;">
                <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error_message); ?>
            </div>
            <a href="mark_sheet_search.php" style="margin-top: 20px; display: inline-block; color: #3498db;">অনুসন্ধান পৃষ্ঠায় ফিরে যান</a>
        </div>
    <?php else: ?>

        <div class="header">
            <h1>মিফতাহুন নাজাত মহিলা মাদরাসা</h1>
            <h2>শিক্ষার্থী তুলনামূলক ফলাফল রিপোর্ট</h2>
        </div>

        <div class="student-info">
            <div>
                <strong>শিক্ষার্থীর নাম:</strong> <?php echo htmlspecialchars($student_data['full_name']); ?>
            </div>
            <div>
                <strong>শ্রেণি:</strong> <?php echo htmlspecialchars($student_data['class_name']); ?>
            </div>
            <div>
                <strong>রোল নং:</strong> <?php echo htmlspecialchars($student_data['class_roll']); ?>
            </div>
            <div>
                <strong>শিক্ষার্থী আইডি:</strong> <?php echo htmlspecialchars($student_data['student_id']); ?>
            </div>
        </div>

        <table>
            <thead>
                <tr>
                    <th rowspan="2" class="subject-col">বিষয়</th>
                    <?php 
                    // প্রতি পরীক্ষার জন্য শিরোনাম তৈরি
                    foreach ($exam_ids as $exam_id): ?>
                        <th colspan="2"><?php echo htmlspecialchars($unique_exams[$exam_id]); ?></th>
                    <?php endforeach; ?>
                    
                    <th rowspan="2" class="analysis-col">পারফরম্যান্স বিশ্লেষণ</th> 
                </tr>
                <tr>
                    <?php 
                    // প্রতি পরীক্ষার জন্য মার্কস ও গ্রেড কলাম তৈরি
                    foreach ($exam_ids as $exam_id): ?>
                        <th class="marks-col">নম্বর</th>
                        <th class="grade-col">গ্রেড</th>
                    <?php endforeach; ?>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($exam_results_grid as $subject_name => $exam_data): ?>
                    <tr>
                        <td class="subject-col"><?php echo $subject_name; ?></td>
                        <?php 
                        $marks_for_analysis = []; // এই অ্যারেতে সব পরীক্ষার নম্বর থাকবে
                        
                        foreach ($exam_ids as $exam_id) {
                            $current_mark = $exam_data[$exam_id]['marks'] ?? null;
                            $grade = $exam_data[$exam_id]['grade'] ?? '—';
                            
                            $marks_display = ($current_mark !== null) ? $current_mark : '—';
                            
                            echo "<td class='marks-col'>$marks_display</td>";
                            echo "<td class='grade-col'>$grade</td>";

                            // বিশ্লেষণের জন্য নম্বর সেভ করা
                            if ($current_mark !== null) {
                                $marks_for_analysis[] = $current_mark;
                            }
                        }
                        ?>
                        
                        <td class="analysis-col">
                            <?php echo analyze_performance($marks_for_analysis); ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <div class="comments-and-signature">
            
            <div class="comment-box">
                <strong>শ্রেণী শিক্ষকের মন্তব্য (Class Teacher's Comment):</strong>
                <div style="height: 50px;"></div>
            </div>

            <div class="comment-box">
                <strong>অভিভাবকের মন্তব্য (Parent's Comment):</strong>
                <div style="height: 50px;"></div>
            </div>
            
            <div class="signature-section">
                <span class="signature-line">
                    স্বাক্ষর (প্রধান শিক্ষক / প্রিন্সিপাল)
                </span>
            </div>
        </div>

    <?php endif; ?>
    </div>
</div>

</body>
</html>