<?php
session_start();
include '../../includes/db_connection.php'; // ডাটাবেস সংযোগ ফাইল
date_default_timezone_set('Asia/Dhaka');

// লগইন এবং অনুমতির যাচাই
if (!isset($_SESSION['user_id']) || ($_SESSION['role'] != 'Admin' && $_SESSION['role'] != 'Teacher')) {
    header("Location: ../../public/index.php"); 
    exit();
}

// ==========================================
// ১. শিক্ষার্থীর আইডি এবং সার্টিফিকেটের তথ্য গ্রহণ
// ==========================================

$student_id = isset($_GET['student_id']) ? intval($_GET['student_id']) : 0;
// calculate_gpa.php থেকে পাস হওয়া তথ্য
$certificate_for = isset($_GET['for']) ? htmlspecialchars($_GET['for']) : 'শ্রেণি শিক্ষায় কৃতিত্ব'; 
$achievement_desc = isset($_GET['desc']) ? htmlspecialchars($_GET['desc']) : 'শ্রেণি শিক্ষায় অসাধারণ ফলাফল অর্জন ও মাদরাসার শৃঙ্খলা বজায় রাখার স্বীকৃতিস্বরূপ';

$student_data = null;
$error_message = '';
$issue_date = date('d F Y', strtotime('now')); // ইস্যুর তারিখ

// বাংলা সংখ্যা ও মাস ম্যাপিং
$bangla_months = array(
    'January' => 'জানুয়ারি', 'February' => 'ফেব্রুয়ারি', 'March' => 'মার্চ',
    'April' => 'এপ্রিল', 'May' => 'মে', 'June' => 'জুন',
    'July' => 'জুলাই', 'August' => 'আগস্ট', 'September' => 'সেপ্টেম্বর',
    'October' => 'অক্টোবর', 'November' => 'নভেম্বর', 'December' => 'ডিসেম্বর'
);
$bangla_numbers = array('১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯', '০');
$english_numbers = array('1', '2', '3', '4', '5', '6', '7', '8', '9', '0');

// ইংরেজি থেকে বাংলায় সংখ্যা রূপান্তরের ফাংশন
function convert_to_bangla_number($number) {
    global $bangla_numbers, $english_numbers;
    // GPA/নম্বর দশমিকসহ থাকতে পারে, তাই শুধু সংখ্যাগুলো পরিবর্তন করা হলো
    return str_replace($english_numbers, $bangla_numbers, $number); 
}

$issue_date_bn = str_replace(array_keys($bangla_months), array_values($bangla_months), $issue_date);
$issue_date_bn = convert_to_bangla_number($issue_date_bn);


// ==========================================
// ২. ডাটাবেস থেকে তথ্য আনা
// ==========================================

if ($student_id > 0) {
    // students টেবিল থেকে প্রয়োজনীয় তথ্য আনা
    $sql = "SELECT 
                s.student_id, 
                s.first_name, 
                s.last_name, 
                s.father_name, 
                c.class_name
            FROM students s
            LEFT JOIN classes c ON s.class_id = c.class_id
            WHERE s.student_id = ?";
    
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $student_id);
        
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            
            if ($result->num_rows === 1) {
                $student_data = $result->fetch_assoc();
                
                // নাম একত্রিত করা
                $student_data['full_name'] = htmlspecialchars($student_data['first_name'] . ' ' . $student_data['last_name']);
                
                // ডেটাবেস সংযোগ বন্ধ করুন
                $stmt->close();
                $conn->close();
                
            } else {
                $error_message = "এই আইডি'র কোনো শিক্ষার্থী পাওয়া যায়নি।";
                $stmt->close();
                $conn->close();
            }
        } else {
            $error_message = "ডাটাবেস ক্যোয়ারি এক্সিকিউট করতে ব্যর্থ: " . $stmt->error;
            $stmt->close();
            $conn->close();
        }
    } else {
        $error_message = "ডাটাবেস ক্যোয়ারি প্রস্তুত করতে ব্যর্থ। ত্রুটি: " . $conn->error;
        $conn->close();
    }
} else {
    $error_message = "সার্টিফিকেট তৈরি করতে একটি সঠিক শিক্ষার্থীর আইডি প্রয়োজন।";
}
?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>সার্টিফিকেট | <?php echo $student_data ? $student_data['full_name'] : 'ত্রুটি'; ?></title>
    <style>
        body {
            font-family: 'Arial', sans-serif;
            background-color: #f4f4f4;
            display: flex;
            flex-direction: column;
            align-items: center;
            padding: 20px;
        }
        .container {
            max-width: 1000px;
            width: 100%;
            background: white;
            padding: 20px;
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.1);
            position: relative;
        }
        /* সার্টিফিকেটের মূল ডিজাইন: ল্যান্ডস্কেপ ওরিয়েন্টেশন */
        .certificate-body {
            border: 10px solid #f39c12; /* বাইরের মোটা বর্ডার */
            border-style: double;
            padding: 50px;
            min-height: 700px; /* প্রিন্ট করার জন্য পর্যাপ্ত উচ্চতা */
            position: relative;
            background-color: #fffaf0; /* অফ-হোয়াইট ব্যাকগ্রাউন্ড */
            text-align: center;
        }
        .header {
            margin-bottom: 20px;
            color: #2c3e50;
        }
        .header h1 {
            font-size: 3em;
            margin: 0 0 5px;
            color: #e67e22; /* উজ্জ্বল রং */
        }
        .header h3 {
            font-size: 1.5em;
            margin: 0;
            font-style: italic;
        }
        .cert-title {
            font-family: 'Times New Roman', serif;
            font-size: 3.5em;
            font-weight: bold;
            color: #007bff; /* সার্টিফিকেটের শিরোনামের রং */
            margin: 40px 0;
            text-shadow: 2px 2px 3px rgba(0,0,0,0.2);
        }
        .cert-subtitle {
            font-size: 1.8em;
            color: #34495e;
            margin-bottom: 15px;
        }
        .student-name {
            font-family: 'Times New Roman', serif;
            font-size: 2.5em;
            font-weight: bold;
            color: #dc3545;
            text-decoration: underline;
            margin: 10px 0;
        }
        .content {
            font-size: 1.3em;
            line-height: 1.8;
            margin-top: 30px;
            padding: 0 50px;
        }
        .content strong {
            color: #000;
        }
        
        /* স্বাক্ষর সেকশন */
        .signature-section {
            display: flex;
            justify-content: space-around;
            margin-top: 80px;
            font-size: 1.2em;
        }
        .signature-item {
            text-align: center;
            width: 30%;
        }
        .signature-line {
            border-top: 2px solid #333;
            padding-top: 10px;
            margin-bottom: 5px;
        }
        .date-info {
            position: absolute;
            bottom: 50px;
            left: 50px;
            font-size: 1.1em;
            font-weight: bold;
            color: #7f8c8d;
        }

        /* প্রিন্ট এবং ব্যাক বাটন স্টাইল */
        .action-buttons {
            margin-bottom: 20px;
            text-align: right;
            width: 1000px;
            max-width: 100%;
        }
        .action-btn {
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-weight: bold;
            margin-left: 10px;
        }
        .print-btn {
            background-color: #2ecc71;
            color: white;
        }
        .back-btn {
            background-color: #f39c12;
            color: white;
            text-decoration: none;
        }

        /* প্রিন্টের জন্য CSS: ল্যান্ডস্কেপ */
        @media print {
            body { 
                background: white; 
                margin: 0; 
                padding: 0;
            }
            .action-buttons { display: none; }
            .container {
                box-shadow: none;
                padding: 0;
                border: none;
                max-width: 100%;
            }
            .certificate-body {
                min-height: 200mm; 
                width: 287mm; 
                margin: 0;
            }
            @page {
                size: A4 landscape; 
                margin: 0.5cm;
            }
        }
    </style>
</head>
<body>

<div class="action-buttons">
    <a href="students_list.php" class="action-btn back-btn">
        ← শিক্ষার্থী তালিকায় ফিরে যান
    </a>
    <button onclick="window.print()" class="action-btn print-btn">
        🖨️ প্রিন্ট করুন (A4 Landscape)
    </button>
</div>

<div class="container">
    <?php if ($student_data): ?>
    <div class="certificate-body">
        <div class="header">
            <h3>মিফতাহুন নাজাত মহিলা মাদরাসা</h3>
            <h1 style="border-bottom: 3px double #e67e22; display: inline-block; padding-bottom: 5px;">কৃতিত্বের সনদপত্র</h1>
            <p>নেয়ামতপুর, সদর, নোয়াখালী।</p>
        </div>

        <div class="cert-title">
            সার্টিফিকেট অফ অ্যাচিভমেন্ট
        </div>
        
        <div class="cert-subtitle">
            এই সনদপত্রটি প্রদান করা হলো
        </div>

        <div class="student-name">
            <?php echo $student_data['full_name']; ?>
        </div>

        <div class="content">
            <p>পিতা: **<?php echo htmlspecialchars($student_data['father_name']); ?>**</p>
            <p>শ্রেণি: **<?php echo htmlspecialchars($student_data['class_name']); ?>**</p>
            
            <p style="margin-top: 30px;">
                তাকে **<?php echo $certificate_for; ?>** এর স্বীকৃতিস্বরূপ এই সনদ প্রদান করা হলো। এই অর্জনটি **<?php echo $achievement_desc; ?>** তার কঠোর পরিশ্রম, একাগ্রতা এবং মাদরাসার প্রতি তার আনুগত্যের প্রমাণ বহন করে।
            </p>
            <p style="margin-top: 20px; font-style: italic; color: #7f8c8d;">
                আমরা তার ভবিষ্যতের জন্য দোয়া করি এবং তার উত্তরোত্তর সাফল্য কামনা করি।
            </p>
        </div>

        <div class="signature-section">
            <div class="signature-item">
                <div class="signature-line"></div>
                <p>শ্রেণি শিক্ষক</p>
            </div>
            <div class="signature-item">
                <div class="signature-line"></div>
                <p>পরিচালনা পর্ষদের সভাপতি</p>
            </div>
            <div class="signature-item">
                <div class="signature-line"></div>
                <p>অধ্যক্ষের স্বাক্ষর ও সীল</p>
            </div>
        </div>
        
        <div class="date-info">
            ইস্যুর তারিখ: <?php echo $issue_date_bn; ?>
        </div>
        
    </div>
    <?php else: ?>
    <div style="text-align: center; color: red; padding: 50px; border: 1px solid red; background-color: #ffe8e8;">
        <h2>ত্রুটি!</h2>
        <p>সার্টিফিকেট তৈরি করা যায়নি। অনুগ্রহ করে নিচের বার্তাটি দেখুন:</p>
        <p style="font-size: 0.9em; font-weight: bold; color: #c0392b;"><?php echo $error_message; ?></p>
        <p>সঠিক আইডি দিয়ে চেষ্টা করুন এবং ডাটাবেস সংযোগ নিশ্চিত করুন।</p>
    </div>
    <?php endif; ?>
</div>

</body>
</html>