<?php
session_start();
// ১. ডাটাবেস সংযোগ পাথের সংশোধন
include '../includes/db_connection.php'; 

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'Admin') {
    header("Location: /school_management/public/index.php");
    exit();
}

if (!$conn) {
    die("ডাটাবেস সংযোগ ব্যর্থ হয়েছে। অনুগ্রহ করে db_connection.php ফাইলটি পরীক্ষা করুন।");
}

$message = '';
$error = '';
$current_year = date('Y');
$next_year = $current_year + 1;

// ==========================================
// A. প্রমোশন প্রক্রিয়া শুরু করা
// ==========================================
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['promote'])) {
    
    $selected_students = $_POST['student_id'] ?? [];
    $target_class_id = $_POST['target_class_id'] ?? null;
    $new_session_year = $_POST['new_session_year'] ?? $next_year; // নতুন সেশন বছর
    $new_roll = 0; // নতুন ক্লাসে রোল নম্বর 0 সেট করা হবে

    // ইনপুট ভ্যালিডেশন
    if (empty($selected_students) || !is_array($selected_students)) {
        $error = "অনুগ্রহ করে প্রমোট করার জন্য কমপক্ষে একজন শিক্ষার্থী নির্বাচন করুন।";
    } elseif (empty($target_class_id) || !is_numeric($target_class_id)) {
        $error = "টার্গেট ক্লাস নির্বাচন করা আবশ্যক এবং তা অবশ্যই বৈধ হতে হবে।";
    } elseif (empty($new_session_year) || !is_numeric($new_session_year)) {
        $error = "নতুন সেশন বছর নির্বাচন করা আবশ্যক।";
    } else {
        $promoted_count = 0;
        $conn->begin_transaction(); // ট্রানজেকশন শুরু

        try {
            // SQL আপডেট কোয়েরি: class_id, session_year, এবং class_roll আপডেট করা হলো
            $update_sql = "UPDATE students SET class_id = ?, session_year = ?, class_roll = ? WHERE student_id = ?";
            $stmt = $conn->prepare($update_sql);
            
            if (!$stmt) {
                 throw new Exception("Prepared statement তৈরি ব্যর্থ: " . $conn->error);
            }
            
            // iisi: class_id(int), session_year(int), class_roll(int), student_id(string/int)
            // আপনার student_id এর ডেটা টাইপ অনুযায়ী শেষ 's' পরিবর্তন করতে হতে পারে। এখানে আমি VARCHAR ধরেছি।
            $stmt->bind_param("iiis", $target_class_id, $new_session_year, $new_roll, $student_id_val);

            foreach ($selected_students as $student_id) {
                $student_id_val = $student_id; 
                if (!$stmt->execute()) {
                    throw new Exception("SQL আপডেট ব্যর্থ: " . $stmt->error);
                }
                $promoted_count++;
            }

            $conn->commit(); // সব আপডেট সফল হলে কমিট
            $message = "সফলভাবে {$promoted_count} জন শিক্ষার্থীকে **{$new_session_year}** সেশনের জন্য নতুন ক্লাসে প্রমোট করা হয়েছে। (সকলের রোল নম্বর রিসেট করা হয়েছে: {$new_roll})";
            $stmt->close();

        } catch (Exception $e) {
            $conn->rollback(); // কোনো সমস্যা হলে রোলব্যাক
            $error = "প্রমোশন প্রক্রিয়া ব্যর্থ হয়েছে। ত্রুটি: " . $e->getMessage();
        }
    }
}

// ==========================================
// B. ক্লাসের তালিকা লোড করা
// ==========================================
$classes_result = $conn->query("SELECT class_id, class_name FROM classes ORDER BY class_id");
$classes = [];
while ($row = $classes_result->fetch_assoc()) {
    $classes[$row['class_id']] = $row['class_name'];
}

// ==========================================
// C. বর্তমান ক্লাসের শিক্ষার্থী তালিকা লোড করা
// ==========================================
$students_to_show = [];
$selected_source_class = $_GET['source_class_id'] ?? null;
$session_filter = $_GET['session_filter'] ?? $current_year; // ড্রপডাউন থেকে সেশন ফিল্টার করা

if ($selected_source_class && is_numeric($selected_source_class) && is_numeric($session_filter)) {
    // নির্বাচিত সেশন এবং ক্লাসের শিক্ষার্থীদের দেখানোর জন্য
    $students_sql = "SELECT s.student_id, s.first_name, s.last_name, c.class_name 
                     FROM students s
                     JOIN classes c ON s.class_id = c.class_id
                     WHERE s.class_id = ? AND s.session_year = ?
                     ORDER BY s.class_roll";
    
    $stmt_students = $conn->prepare($students_sql);
    $stmt_students->bind_param("ii", $selected_source_class, $session_filter);
    $stmt_students->execute();
    $students_result = $stmt_students->get_result();
    
    while ($row = $students_result->fetch_assoc()) {
        $students_to_show[] = $row;
    }
    $stmt_students->close();
}
// যেহেতু $conn এর কানেকশন উপরে বন্ধ করা হয়েছে, তাই এখানে পুনরায় বন্ধ করার প্রয়োজন নেই।

// ==========================================
// D. সেশন ইয়ার ড্রপডাউন তৈরি করার জন্য ফাংশন
// ==========================================
function get_session_years($startYear = 2020) {
    $currentYear = date('Y') + 1;
    $years = [];
    for ($i = $currentYear; $i >= $startYear; $i--) {
        $years[] = $i;
    }
    return $years;
}
?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>শিক্ষার্থী প্রমোশন - <?php echo $current_year; ?> থেকে <?php echo $next_year; ?></title>
    <link rel="stylesheet" href="css/style.css"> 
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        .promotion-container { max-width: 1000px; margin: 20px auto; padding: 20px; background-color: #fff; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
        .filter-form, .promotion-controls { background-color: #f7f9fc; padding: 15px; border-radius: 5px; margin-bottom: 20px; border: 1px solid #eee; }
        .action-button { background-color: #27ae60; color: white; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; font-weight: bold; }
        .action-button:hover { background-color: #229954; }
        .student-table { width: 100%; border-collapse: collapse; margin-top: 15px; }
        .student-table th, .student-table td { border: 1px solid #ddd; padding: 10px; text-align: left; }
    </style>
</head>
<body>
    <div class="header">
        <h1>শিক্ষার্থী প্রমোশন</h1>
        <a href="dashboard.php" class="button" style="background-color: #3498db;">ড্যাশবোর্ডে ফিরে যান</a>
    </div>

    <div class="content">
        <div class="promotion-container">
            <?php if ($message): ?>
                <p style='color:green; background-color: #d4edda; padding: 10px; border-radius: 4px;'><?php echo htmlspecialchars($message); ?></p>
            <?php endif; ?>
            <?php if ($error): ?>
                <p style='color:red; background-color: #f8d7da; padding: 10px; border-radius: 4px;'><?php echo htmlspecialchars($error); ?></p>
            <?php endif; ?>

            <div class="filter-form">
                <form method="GET" action="promote_students.php" style="display: flex; gap: 15px; align-items: center;">
                    <div>
                        <label for="session_filter">সেশন ইয়ার:</label>
                        <select name="session_filter" id="session_filter" required>
                            <?php foreach (get_session_years() as $year): ?>
                                <option value="<?php echo $year; ?>" 
                                    <?php echo ($session_filter == $year) ? 'selected' : ''; ?>>
                                    <?php echo $year; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <label for="source_class_id">ক্লাস:</label>
                        <select name="source_class_id" id="source_class_id" required>
                            <option value="">ক্লাস নির্বাচন করুন</option>
                            <?php foreach ($classes as $id => $name): ?>
                                <option value="<?php echo $id; ?>" 
                                    <?php echo ($selected_source_class == $id) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($name); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <button type="submit" class="action-button" style="background-color: #3498db;">শিক্ষার্থী দেখুন</button>
                </form>
            </div>

            <?php if ($selected_source_class && empty($students_to_show) && empty($error) && empty($message)): ?>
                <p style="padding: 15px; background-color: #fff3cd; border: 1px solid #ffeeba; color: #856404;">
                    <?php echo htmlspecialchars($classes[$selected_source_class]); ?> ক্লাসে (<?php echo htmlspecialchars($session_filter); ?> সেশনের) কোনো শিক্ষার্থী খুঁজে পাওয়া যায়নি।
                </p>
            <?php endif; ?>

            <?php if (!empty($students_to_show)): ?>
                <form method="POST" action="promote_students.php">
                    <div class="promotion-controls">
                        <h3>প্রমোশন সেটিং:</h3>
                        <div style="display: flex; gap: 20px; align-items: center; flex-wrap: wrap;">
                            
                            <div>
                                <label for="new_session_year">**নতুন সেশন:**</label>
                                <select name="new_session_year" id="new_session_year" required>
                                    <?php foreach (get_session_years() as $year): ?>
                                        <option value="<?php echo $year; ?>" 
                                            <?php echo ($year == $next_year) ? 'selected' : ''; ?>>
                                            <?php echo $year; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div>
                                <label for="target_class_id">**টার্গেট ক্লাস নির্বাচন করুন:**</label>
                                <select name="target_class_id" id="target_class_id" required>
                                    <option value="">পরবর্তী ক্লাস</option>
                                    <?php 
                                    $next_class_id = $selected_source_class + 1; 
                                    foreach ($classes as $id => $name): 
                                        if ($id > $selected_source_class):
                                    ?>
                                        <option value="<?php echo $id; ?>" <?php echo ($id == $next_class_id) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($name); ?>
                                        </option>
                                    <?php 
                                        endif;
                                    endforeach; ?>
                                </select>
                            </div>
                            <button type="submit" name="promote" class="action-button">
                                <i class="fas fa-level-up-alt"></i> নির্বাচিতদের প্রমোট করুন
                            </button>
                        </div>
                    </div>

                    <table class="student-table">
                        <thead>
                            <tr>
                                <th><input type="checkbox" id="selectAll" checked> সব নির্বাচন</th>
                                <th>আইডি</th>
                                <th>শিক্ষার্থীর নাম</th>
                                <th>বর্তমান ক্লাস</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($students_to_show as $student): ?>
                                <tr>
                                    <td>
                                        <input type="checkbox" name="student_id[]" class="student-checkbox" value="<?php echo htmlspecialchars($student['student_id']); ?>" checked>
                                    </td>
                                    <td><?php echo htmlspecialchars($student['student_id']); ?></td>
                                    <td><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></td>
                                    <td><?php echo htmlspecialchars($student['class_name']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </form>
            <?php endif; ?>
        </div>
    </div>
<script>
    document.getElementById('selectAll').onclick = function () {
        var checkboxes = document.querySelectorAll('.student-checkbox');
        for (var i = 0; i < checkboxes.length; i++) {
            checkboxes[i].checked = this.checked;
        }
    };
</script>
</body>
</html>