<?php
// সেশন শুরু করা যদি প্রয়োজন হয়, যদিও ফি রিপোর্টের জন্য আবশ্যক নয়, তবে সুরক্ষা ও নেভিগেশনের জন্য ভালো।
// session_start(); 

include '../includes/db_connection.php';

// ইনপুট প্যারামিটার গ্রহণ
$selected_month = isset($_GET['month']) ? $_GET['month'] : date('Y-m');
$search_query = isset($_GET['search']) ? trim($_GET['search']) : ''; // নতুন সার্চ ক্যোয়ারি

// ==========================================
// ১. মূল SQL কোয়েরি তৈরি
// ==========================================
$sql = "SELECT 
            s.first_name, 
            s.last_name, 
            s.student_id,
            f.amount, 
            f.payment_date 
        FROM fees f
        JOIN students s ON f.student_id = s.student_id
        WHERE DATE_FORMAT(f.payment_month, '%Y-%m') = ?";

// সার্চ ক্লজ যোগ করা
$bind_params = "s"; // প্রথম প্যারামিটারটি মাস (string)
$bind_values = [$selected_month];

if (!empty($search_query)) {
    $search_term = "%" . $search_query . "%";
    
    // শিক্ষার্থীর নাম অথবা আইডি দ্বারা সার্চ করা হচ্ছে
    $sql .= " AND (s.first_name LIKE ? OR s.last_name LIKE ? OR s.student_id = ?)";
    
    // প্যারামিটার বাইন্ডিং এর প্রকার (types) যোগ করা
    $bind_params .= "ssi"; 
    
    // প্যারামিটারের মান (values) যোগ করা
    $bind_values[] = $search_term;
    $bind_values[] = $search_term;
    // যদি সার্চ করা শব্দ সংখ্যা হয় তবে student_id হিসেবে যোগ করা, অন্যথায় 0
    $bind_values[] = is_numeric($search_query) ? (int)$search_query : 0; 
}

$sql .= " ORDER BY f.payment_date DESC";

// ==========================================
// ২. Prepared Statement কার্যকর করা
// ==========================================
$stmt = $conn->prepare($sql);

if ($stmt === false) {
    die("Statement preparation failed: " . $conn->error);
}

// bind_param ডায়নামিকভাবে কল করা
// (এই সিনট্যাক্সটি PHP 5.6+ এ কাজ করবে, যদি আপনার সার্ভার পুরাতন হয় তবে এটি ম্যানুয়ালি পরিবর্তন করতে হতে পারে।)
$stmt->bind_param($bind_params, ...$bind_values);
$stmt->execute();
$result = $stmt->get_result();

$total_amount = 0; // মোট সংগ্রহের জন্য ভেরিয়েবল

?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>মাসিক ফি রিপোর্ট</title>
    <link rel="stylesheet" href="../css/style.css"> 
    <style>
        /* CSS স্টাইল */
        body { font-family: Arial, sans-serif; }
        h2, h3 { color: #333; }
        .container {
            max-width: 900px;
            margin: 20px auto;
            padding: 20px;
            background-color: #fff;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
            border-radius: 8px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 10px;
            text-align: left;
            font-size: 14px;
        }
        th {
            background-color: #f2f2f2;
            color: #333;
        }
        /* সার্চ ফর্মের স্টাইল */
        .search-form-container {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            align-items: center;
            flex-wrap: wrap; /* ছোট স্ক্রিনের জন্য */
        }
        .search-form-container input[type="month"],
        .search-form-container input[type="text"],
        .search-form-container button,
        .search-form-container a.button {
            padding: 8px 12px;
            border: 1px solid #ccc;
            border-radius: 4px;
            font-size: 14px;
            text-decoration: none;
            text-align: center;
        }
        .search-form-container button {
            background-color: #3498db;
            color: white;
            cursor: pointer;
            border: none;
        }
        .search-form-container button:hover {
            background-color: #2980b9;
        }
        .search-form-container a.button {
             background-color: #f39c12; 
             color: white;
        }
    </style>
</head>
<body>
    <div class="container">
        <h2>মাসিক ফি রিপোর্ট</h2>
        
        <form action="" method="GET" class="search-form-container">
            <label for="month" style="font-weight: bold;">মাস:</label>
            <input 
                type="month" 
                id="month" 
                name="month" 
                value="<?php echo htmlspecialchars($selected_month); ?>"
            >
            <input 
                type="text" 
                name="search" 
                placeholder="শিক্ষার্থীর নাম/আইডি দিয়ে খুঁজুন..." 
                value="<?php echo htmlspecialchars($search_query); ?>"
                style="flex-grow: 1; min-width: 250px;"
            >
            <button type="submit">রিপোর্ট দেখুন</button>

            <?php if (!empty($search_query) || $selected_month != date('Y-m')): ?>
                <a href="?month=<?php echo date('Y-m'); ?>" class="button">ফিল্টার মুছুন</a>
            <?php endif; ?>
        </form>

        <?php
        $month_name = date('F Y', strtotime($selected_month));
        echo "<h3 style='margin-bottom: 10px;'>রিপোর্ট মাস: " . htmlspecialchars($month_name) . "</h3>";

        if ($result->num_rows > 0) {
            echo "<table>";
            echo "<thead><tr><th>শিক্ষার্থী আইডি</th><th>শিক্ষার্থীর নাম</th><th>জমা হওয়া টাকা (৳)</th><th>পেমেন্টের তারিখ</th></tr></thead>";
            echo "<tbody>";
            
            while($row = $result->fetch_assoc()) {
                echo "<tr>";
                echo "<td>" . htmlspecialchars($row['student_id']) . "</td>";
                echo "<td>" . htmlspecialchars($row['first_name'] . ' ' . $row['last_name']) . "</td>";
                // টাকার পরিমাণ বাংলাদেশি ফরম্যাটে দেখানো
                echo "<td>" . htmlspecialchars(number_format($row['amount'], 0)) . "</td>";
                // তারিখ ফরম্যাট করা
                echo "<td>" . htmlspecialchars(date('d-m-Y', strtotime($row['payment_date']))) . "</td>";
                echo "</tr>";
                $total_amount += $row['amount'];
            }
            echo "</tbody></table>";
            echo "<h3 style='text-align: right; margin-top: 15px; color: #28a745;'>মোট সংগ্রহ: " . number_format($total_amount, 0) . " BDT</h3>";
        } else {
            echo "<p>এই মাসে কোনো ফি পেমেন্ট পাওয়া যায়নি" . (!empty($search_query) ? " অথবা খোঁজা শিক্ষার্থীর ডেটা নেই।" : ".") . "</p>";
        }
        
        $stmt->close();
        $conn->close();
        ?>
    </div>
</body>
</html>