<?php
// edit_fee.php

session_start();
include '../../includes/db_connection.php'; // ডাটাবেস সংযোগ নিশ্চিত করা
include '../../includes/functions.php';

// নিশ্চিত করা যে ব্যবহারকারী লগইন করেছেন এবং তাদের অনুমতি আছে
if (!isset($_SESSION['user_id']) || ($_SESSION['role'] !== 'Admin' && $_SESSION['role'] !== 'Accountant')) {
    header("Location: /school_management/public/index.php");
    exit();
}

$message = '';
$error = '';
$fee_data = null;
$months = [
    '01' => 'জানুয়ারি', '02' => 'ফেব্রুয়ারি', '03' => 'মার্চ',
    '04' => 'এপ্রিল', '05' => 'মে', '06' => 'জুন',
    '07' => 'জুলাই', '08' => 'আগস্ট', '09' => 'সেপ্টেম্বর',
    '10' => 'অক্টোবর', '11' => 'নভেম্বর', '12' => 'ডিসেম্বর'
];
$currentYear = date('Y');
$years = range($currentYear, $currentYear - 5);


// ----------------------------------------------------------------
// A. পেমেন্ট আইডি দিয়ে বর্তমান ডেটা লোড করা
// ----------------------------------------------------------------
if (isset($_GET['id'])) {
    $payment_id = $_GET['id'];
    
    // শিক্ষার্থীর নাম, শ্রেণি এবং ফি-এর ক্যাটাগরি সহ ডেটা লোড করার SQL
    $sql = "SELECT f.*, s.first_name, s.last_name, c.class_name, ic.category_name 
            FROM fees f
            JOIN students s ON f.student_id = s.student_id
            JOIN classes c ON s.class_id = c.class_id
            JOIN income_categories ic ON f.category_id = ic.category_id
            WHERE f.payment_id = ?";

    $stmt = $conn->prepare($sql);
    
    if ($stmt === false) {
        // prepare() ব্যর্থ হলে ত্রুটি দেখানো
        $error = "ডাটা লোডের জন্য SQL ক্যোয়ারি তৈরিতে সমস্যা: " . $conn->error;
    } else {
        $stmt->bind_param("i", $payment_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $fee_data = $result->fetch_assoc();
        $stmt->close();

        if (!$fee_data) {
            $error = "এই আইডি সহ কোনো ফি রেকর্ড খুঁজে পাওয়া যায়নি।";
        }
    }
} else if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    $error = "কোনো পেমেন্ট আইডি সরবরাহ করা হয়নি।";
}


// ----------------------------------------------------------------
// B. ফর্ম সাবমিট হলে ডেটা আপডেট করা
// ----------------------------------------------------------------
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['payment_id'])) {
    $payment_id = $_POST['payment_id'];
    $amount = filter_input(INPUT_POST, 'amount', FILTER_VALIDATE_FLOAT);
    $payment_date = $_POST['payment_date'];
    $payment_method = sanitize_input($_POST['payment_method']);
    $fee_month = sanitize_input($_POST['fee_month']);
    $memo_no = sanitize_input($_POST['memo_no']);

    if ($amount === false || $amount <= 0 || empty($payment_date) || empty($payment_method)) {
        $error = "ফর্মের তথ্য ভুল বা অসম্পূর্ণ। দয়া করে সঠিক তথ্য দিন।";
    } else {
        // SQL ক্যোয়ারি: এই লাইনেই সম্ভবত আপনার আগের ভুলটি ছিল।
        // নিশ্চিত করুন যে কলামের নামগুলো (amount, payment_date, payment_method, fee_month, memo_no) সঠিক।
        $sql = "UPDATE fees SET amount = ?, payment_date = ?, payment_method = ?, fee_month = ?, memo_no = ? WHERE payment_id = ?";
        
        $stmt = $conn->prepare($sql);
        
        // এখানে prepare() ফাংশন ব্যর্থ হলে $stmt এর মান 'false' হবে।
        if ($stmt === false) {
             // **এই ব্লকটি আপনার Fatal Error সমাধান করে**
             $error = "আপডেট SQL ক্যোয়ারি তৈরিতে সমস্যা। সম্ভবত কলামের নাম ভুল আছে। ত্রুটি: " . $conn->error;
        } else {
            // **লাইন 45: bind_param() সফলভাবে কল হবে**
            $stmt->bind_param("dssssi", $amount, $payment_date, $payment_method, $fee_month, $memo_no, $payment_id);
            
            if ($stmt->execute()) {
                $message = "ফি রেকর্ড সফলভাবে আপডেট করা হয়েছে!";
                // সফলভাবে আপডেটের পর ডেটা আবার লোড করা 
                header("Location: edit_fee.php?id=" . $payment_id . "&success=" . urlencode($message));
                exit();
            } else {
                $error = "ফি আপডেট করার সময় সমস্যা হয়েছে: " . $stmt->error;
            }
            $stmt->close();
        }
    }
}

// সফল মেসেজ যদি রিডাইরেক্ট করে আসে
if (isset($_GET['success'])) {
    $message = sanitize_input($_GET['success']);
}

// ক্যাটাগরি এবং পেমেন্ট মেথড লোড করা (ড্রপডাউনের জন্য)
$categories_sql = "SELECT category_id, category_name FROM income_categories";
$categories_result = $conn->query($categories_sql);
$payment_methods = ['Cash', 'Bank', 'Mobile Banking (Bkash/Nagad)'];
?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ফি রেকর্ড সম্পাদনা</title>
    <link rel="stylesheet" href="/school_management/public/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
</head>
<body>
<div class="header">
    <h1>স্কুল ম্যানেজমেন্ট সিস্টেম</h1>
    <div class="nav">
        <a href="../dashboard.php">ড্যাশবোর্ড</a>
        <a href="fee_report.php">ফি রিপোর্ট</a>
        <a href="fee_collection.php">ফি সংগ্রহ</a>
        <a href="../logout.php">লগআউট</a>
    </div>
</div>
<div class="content">
    <div class="container">
        <h2>ফি রেকর্ড সম্পাদনা</h2>
        
        <p><a href="fee_report.php" class="button"><i class="fas fa-arrow-left"></i> ফি রিপোর্টে ফিরে যান</a></p>

        <?php if ($message): ?>
            <p style='color:green; font-weight: bold;'><?php echo $message; ?></p>
        <?php endif; ?>
        <?php if ($error): ?>
            <p style='color:red; font-weight: bold;'><?php echo $error; ?></p>
        <?php endif; ?>

        <?php if ($fee_data): ?>
            <div class="fee-info-box" style="border: 1px solid #ccc; padding: 15px; margin-bottom: 20px; background-color: #f9f9f9;">
                <p><strong>শিক্ষার্থীর আইডি:</strong> <?php echo htmlspecialchars($fee_data['student_id']); ?></p>
                <p><strong>শিক্ষার্থীর নাম:</strong> <?php echo htmlspecialchars($fee_data['first_name'] . ' ' . $fee_data['last_name']); ?></p>
                <p><strong>শ্রেণি:</strong> <?php echo htmlspecialchars($fee_data['class_name']); ?></p>
                <p><strong>ফি-এর ধরন:</strong> <?php echo htmlspecialchars($fee_data['category_name']); ?></p>
            </div>
            
            <form method="POST" action="">
                <input type="hidden" name="payment_id" value="<?php echo htmlspecialchars($fee_data['payment_id']); ?>">
                
                <div class="form-group">
                    <label for="amount">জমা করা ফি (টাকা):</label>
                    <input type="number" id="amount" name="amount" value="<?php echo htmlspecialchars($fee_data['amount']); ?>" required min="1">
                </div>
                
                <div class="form-group">
                    <label for="payment_date">জমা দেওয়ার তারিখ:</label>
                    <input type="date" id="payment_date" name="payment_date" value="<?php echo htmlspecialchars($fee_data['payment_date']); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="payment_method">পেমেন্টের মাধ্যম:</label>
                    <select id="payment_method" name="payment_method" required>
                        <?php foreach ($payment_methods as $method): ?>
                            <option value="<?php echo htmlspecialchars($method); ?>" <?php echo ($fee_data['payment_method'] === $method) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($method); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="fee_month">মাস (মাসিক ফি-এর জন্য):</label>
                    <?php 
                        $current_fee_month = $fee_data['fee_month'] ?? '';
                    ?>
                    <select id="fee_month" name="fee_month">
                        <option value="">নির্বাচন করুন (ঐচ্ছিক)</option>
                        <?php 
                            foreach ($years as $year) {
                                foreach ($months as $num => $name) {
                                    $fullMonth = $year . '-' . $num;
                                    $selected = ($current_fee_month === $fullMonth) ? 'selected' : '';
                                    echo "<option value='" . htmlspecialchars($fullMonth) . "' $selected>" . htmlspecialchars($name) . " " . htmlspecialchars($year) . "</option>";
                                }
                            }
                        ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="memo_no">মেমো নং:</label>
                    <input type="text" id="memo_no" name="memo_no" value="<?php echo htmlspecialchars($fee_data['memo_no'] ?? ''); ?>">
                </div>

                <button type="submit" class="submit-btn"><i class="fas fa-save"></i> আপডেট করুন</button>
            </form>
        <?php else: ?>
            <p style='color:red;'>রেকর্ড সম্পাদনার জন্য ডেটা লোড করা সম্ভব হয়নি।</p>
        <?php endif; ?>
    </div>
</div>
<div class="footer">
    <p>&copy; <?php echo date("Y"); ?> স্কুল ম্যানেজমেন্ট সিস্টেম। সর্বস্বত্ব সংরক্ষিত।</p>
</div>
</body>
</html>

<?php
$conn->close();
?>