<?php
// এই ফাইলটি expense_report.php এর পপ-আপের জন্য ডেটা প্রদান করবে।
session_start();
header('Content-Type: application/json');

// পাথের সঠিকতা নিশ্চিত করুন
include '../../includes/db_connection.php'; 

$response = [
    'error' => '', 
    'voucher_no' => '', 
    'expense_date' => '', 
    'paid_to' => '', 
    'main_description' => '', 
    'received_by_name' => '', // যিনি এন্ট্রি করেছেন
    // 'teacher_name' এবং 'salary_month' ফিল্ড দুটি বাদ দেওয়া হলো
    'details' => [], 
    'total_amount' => '০.০০'
];

// বাংলা সংখ্যা ও তারিখ রূপান্তর ফাংশন
function format_to_bangla_number($number) {
    return str_replace(range(0, 9), array('০', '১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯'), number_format($number, 2));
}

function format_date_to_bangla($date_string) {
    if (empty($date_string)) return '';
    $timestamp = strtotime($date_string);
    $formatted_date = date('d-m-Y', $timestamp);
    return str_replace(range(0, 9), array('০', '১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯'), $formatted_date);
}

// মাসের নাম বাংলায়
function format_month_to_bangla($month_year) {
    if (empty($month_year)) return '';
    $timestamp = strtotime($month_year . '-01');
    $year = format_date_to_bangla(date('Y', $timestamp));
    $month_num = (int)date('m', $timestamp);

    $bangla_months = [
        1 => 'জানুয়ারি', 2 => 'ফেব্রুয়ারি', 3 => 'মার্চ', 4 => 'এপ্রিল', 
        5 => 'মে', 6 => 'জুন', 7 => 'জুলাই', 8 => 'আগস্ট', 
        9 => 'সেপ্টেম্বর', 10 => 'অক্টোবর', 11 => 'নভেম্বর', 12 => 'ডিসেম্বর'
    ];
    return $bangla_months[$month_num] . ' ' . $year;
}


// ইনপুট ভ্যালিডেশন
if (!isset($_GET['voucher_no']) || empty($_GET['voucher_no'])) {
    $response['error'] = "ভাউচার নং নির্দিষ্ট করা হয়নি।";
    echo json_encode($response);
    exit();
}

$voucher_no = $_GET['voucher_no'];

if (!isset($conn) || $conn->connect_error) {
    $response['error'] = "ডাটাবেস সংযোগ ব্যর্থ: " . (isset($conn) ? $conn->connect_error : "সংযোগ পাওয়া যায়নি");
    echo json_encode($response);
    exit();
}

// === ভাউচারের বিস্তারিত তথ্য আনার জন্য কোয়েরি ===
$sql = "
    SELECT 
        e.expense_date,
        e.amount,
        e.description AS entry_description,
        e.paid_to,
        e.received_by,
        e.teacher_id,
        e.salary_month,
        c.category_name,
        u.username AS received_by_name,
        CONCAT(t.first_name, ' ', t.last_name) AS teacher_name
    FROM expenses e
    JOIN expense_categories c ON e.category_id = c.category_id
    LEFT JOIN users u ON e.received_by = u.user_id
    LEFT JOIN teachers t ON e.teacher_id = t.teacher_id
    WHERE e.voucher_no = ?
    ORDER BY e.teacher_id DESC, e.expense_date ASC, c.category_name ASC
";

$stmt = $conn->prepare($sql);

if ($stmt === false) {
    $response['error'] = "ডাটাবেস প্রস্তুতি ব্যর্থ: " . $conn->error;
} else {
    $stmt->bind_param("s", $voucher_no);
    
    if ($stmt->execute()) {
        $result = $stmt->get_result();
        $details = [];
        $total_amount = 0;
        $serial = 1;
        $first_row = true;
        
        while ($row = $result->fetch_assoc()) {
            
            // ভাউচারের সাধারণ তথ্যগুলো প্রথম রো থেকে নেওয়া হবে
            if ($first_row) {
                $response['voucher_no'] = htmlspecialchars($voucher_no);
                $response['expense_date'] = format_date_to_bangla($row['expense_date']);
                $response['paid_to'] = htmlspecialchars($row['paid_to']); 
                $response['received_by_name'] = htmlspecialchars($row['received_by_name']); 
                $response['main_description'] = htmlspecialchars($row['entry_description']); 
                $first_row = false;
            }
            
            $amount_float = (float)$row['amount'];
            $total_amount += $amount_float;

            $entry_description_html = htmlspecialchars($row['entry_description']); // অন্যান্য খরচের বিবরণ

            if (!empty($row['teacher_id'])) {
                $salary_month_clean = substr($row['salary_month'], 0, 7); 
                $formatted_month = format_month_to_bangla($salary_month_clean);
                
                // বিবরণ কলামের জন্য কাস্টম HTML তৈরি
                $entry_description_html = '
                    <div style="font-weight: bold; color: #3498db; margin-bottom: 3px;">
                        শিক্ষকের নাম: ' . htmlspecialchars($row['teacher_name']) . '
                    </div>
                    <div style="color: #2ecc71;">
                        বেতন মাস: ' . $formatted_month . '
                    </div>
                ';
            }
            
            // লেনদেনের বিস্তারিত তথ্য
            $details[] = [
                'serial' => str_replace(range(0, 9), array('০', '১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯'), $serial++), // বাংলা ক্রমিক
                'category_name' => htmlspecialchars($row['category_name']),
                // বিবরণ কলামে শিক্ষকের তথ্য অথবা অন্য বিবরণ থাকবে
                'entry_description' => $entry_description_html, 
                'amount' => format_to_bangla_number($amount_float),
            ];
        }
        
        if ($result->num_rows > 0) {
            $response['details'] = $details;
            $response['total_amount'] = format_to_bangla_number($total_amount);
        } else {
            $response['error'] = "এই ভাউচার নং এর জন্য কোনো বিস্তারিত এন্ট্রি পাওয়া যায়নি।";
        }
    } else {
        $response['error'] = "কোয়েরি এক্সিকিউশনে সমস্যা: " . $stmt->error;
    }
    
    $stmt->close();
}

if (isset($conn) && $conn->ping()) {
    $conn->close();
}

echo json_encode($response);