<?php
session_start();
// পাথের সঠিকতা নিশ্চিত করুন
include '../../includes/db_connection.php'; 

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'Admin') {
    die("আপনার এই পাতা ব্যবহারের অনুমতি নেই।");
}

// === তারিখ ফিল্টারের জন্য ভ্যারিয়েবল সেট করা ===
$start_date = $_GET['start_date'] ?? '';
$end_date = $_GET['end_date'] ?? '';

// ===========================================
// ১. ব্যয়ের খাত (Category) এবং তাদের ID সংগ্রহ
// ===========================================
$categories = [];
$category_ids = [];
$category_sql = "SELECT category_id, category_name FROM expense_categories WHERE is_active = TRUE ORDER BY category_name ASC";
$category_result = $conn->query($category_sql);

if ($category_result) {
    while ($row = $category_result->fetch_assoc()) {
        $categories[$row['category_id']] = $row['category_name'];
        $category_ids[] = $row['category_id'];
    }
}

// ===========================================
// ২. ভাউচার ভিত্তিক ডেটা প্রস্তুত করা
// ===========================================
$voucher_expenses = [];
$grand_total_expenses = 0;
$column_totals = array_fill_keys($category_ids, 0.00);

// --- তারিখ ফিল্টার লজিক যুক্ত করা ---
$where_clause = "";
$param_types = "";
$param_values = [];

if (!empty($start_date) && !empty($end_date)) {
    $where_clause = " WHERE e.expense_date BETWEEN ? AND ?";
    $param_types = "ss";
    $param_values = [$start_date, $end_date];
}

// সব ব্যয় লেনদেন লোড করা এবং ভাউচার ও খাত অনুযায়ী গ্রুপ করা
// MAX(e.description) ব্যবহার করা হয়েছে কারণ আমরা GROUP BY করছি
$expense_sql = "
    SELECT 
        e.voucher_no,
        MAX(e.expense_date) AS expense_date,
        e.category_id,
        MAX(e.description) AS description,
        SUM(e.amount) AS total_amount
    FROM expenses e
    " . $where_clause . "
    GROUP BY e.voucher_no, e.category_id
    ORDER BY expense_date DESC, e.voucher_no ASC
";

$stmt = $conn->prepare($expense_sql);

if (!empty($param_values)) {
    // প্যারামিটার বাইন্ডিং
    $stmt->bind_param($param_types, ...$param_values);
}

$result = false;
if ($stmt->execute()) {
    $result = $stmt->get_result();
}

if ($result) {
    while ($row = $result->fetch_assoc()) {
        $voucher = $row['voucher_no'];
        $cat_id = $row['category_id'];
        $amount = (float)$row['total_amount'];
        $date = $row['expense_date'];
        $desc = $row['description'];

        // ভাউচার ভিত্তিক কাঠামো তৈরি
        if (!isset($voucher_expenses[$voucher])) {
            $voucher_expenses[$voucher] = [
                'date' => $date,
                'description' => $desc,
                'categories' => array_fill_keys($category_ids, 0.00),
                'voucher_total' => 0.00
            ];
        }

        // খাত অনুযায়ী পরিমাণ যোগ করা
        if (array_key_exists($cat_id, $voucher_expenses[$voucher]['categories'])) {
            $voucher_expenses[$voucher]['categories'][$cat_id] = $amount;
            $voucher_expenses[$voucher]['voucher_total'] += $amount;
        }
    }
    
    // গ্র্যান্ড টোটাল ও কলাম টোটাল হিসেব করা
    foreach ($voucher_expenses as $data) {
        $grand_total_expenses += $data['voucher_total'];
        foreach ($category_ids as $cat_id) {
            $column_totals[$cat_id] += $data['categories'][$cat_id];
        }
    }
}
$stmt->close();

// ===========================================
// ৩. বাংলা সংখ্যা ও তারিখ রূপান্তর ফাংশন
// ===========================================
function format_to_bangla_number($number) {
    return str_replace(range(0, 9), array('০', '১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯'), number_format($number, 2));
}

function format_date_to_bangla($date_string) {
    if (empty($date_string)) return '';
    $timestamp = strtotime($date_string);
    $formatted_date = date('d-m-Y', $timestamp);
    return str_replace(range(0, 9), array('০', '১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯'), $formatted_date);
}

// চূড়ান্ত ডাটাবেস সংযোগ বন্ধ করা
if (isset($conn) && $conn->ping()) {
    $conn->close();
}
?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ব্যয় রিপোর্ট (ভাউচার ভিত্তিক)</title>
    <link rel="stylesheet" href="../../public/css/style.css"> 
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        /* Base Styles for Report (অপরিবর্তিত) */
        .header { margin-bottom: 0; }
        .report-container { 
            max-width: 1300px; 
            margin: 0 auto; 
            padding: 30px; 
            background-color: #fff; 
            border-radius: 8px; 
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
            position: relative;
        }
        .report-container h2 { text-align: center; color: #34495e; margin-bottom: 20px; }
        
        /* ব্যাক বাটন ও প্রিন্ট বাটন */
        .control-group { 
            display: flex; 
            justify-content: space-between; 
            align-items: center; 
            margin-bottom: 20px; 
        }
        .back-button, .print-button, .expense-button {
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            font-weight: bold;
            display: inline-flex;
            align-items: center;
            gap: 5px;
            transition: background-color 0.3s;
        }
        .back-button { 
            background-color: #34495e; color: white; border: none; 
            position: absolute; top: 20px; left: 20px; 
        }
        .print-button { background-color: #2980b9; color: white; border: none; }
        .expense-button { background-color: #2ecc71; color: white; border: none; }
        .back-button:hover { background-color: #2c3e50; }
        .print-button:hover { background-color: #1a6f9c; }
        .expense-button:hover { background-color: #27ae60; }

        /* সার্চ ফর্ম স্টাইল */
        .search-form {
            display: flex;
            gap: 10px;
            align-items: center;
            justify-content: center;
            margin-bottom: 20px;
            padding: 15px;
            background-color: #f4f7f9;
            border-radius: 8px;
        }
        .search-form input[type="date"], .search-form button {
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 4px;
            font-size: 16px;
        }
        .search-form button {
            background-color: #f39c12;
            color: white;
            cursor: pointer;
            border: none;
        }
        .search-form button:hover { background-color: #d35400; }
        
        /* টেবিল স্টাইল */
        .report-table { width: 100%; border-collapse: collapse; margin-top: 20px; font-size: 0.95em; overflow-x: auto; display: block; }
        .report-table th, .report-table td { border: 1px solid #ddd; padding: 12px 8px; text-align: right; white-space: nowrap; }
        .report-table th { background-color: #34495e; color: white; text-align: center; }
        .report-table .fixed-col { text-align: left; width: 100px; }
        .report-table .desc-col { text-align: left; min-width: 180px; }
        .report-table tfoot th { background-color: #e0f7fa; font-weight: bold; }
        .report-table .total-col { background-color: #ffecb3; font-weight: bold; }
        .report-table .grand-total-row th { background-color: #ff9800; color: white; font-size: 1.1em; }
        
        /* অ্যাকশন বাটন */
        .action-link { 
            text-decoration: none; padding: 6px 10px; border-radius: 4px; 
            font-weight: bold; font-size: 0.85em; transition: background-color 0.3s; 
            display: inline-flex; align-items: center; gap: 3px; margin: 2px;
        }
        .edit-btn { background-color: #007bff; color: white; }
        .delete-btn { background-color: #dc3545; color: white; }
        .edit-btn:hover { background-color: #0056b3; }
        .delete-btn:hover { background-color: #c82333; }

        /* প্রিন্ট হেডার স্টাইল (নন-প্রিন্ট মোডে লুকানো) */
        .school-header-print {
            display: none; 
            text-align: center;
            margin-bottom: 20px;
        }
        .school-header-print h3, .school-header-print p, .school-header-print h4 {
            margin: 5px 0;
            color: #000;
        }

        /* --- নতুন পপ-আপ (Modal) স্টাইল শুরু --- */
        .modal {
            display: none; /* লুকানো */
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.5);
            padding-top: 60px;
        }
        .modal-content {
            background-color: #fefefe;
            margin: 5% auto;
            padding: 25px;
            border: 1px solid #888;
            width: 90%;
            max-width: 700px; /* পপ-আপের আকার বাড়ানো হয়েছে */
            border-radius: 10px;
            position: relative;
            box-shadow: 0 5px 15px rgba(0,0,0,0.3);
            animation: fadeIn 0.3s;
        }
        @keyframes fadeIn {
            from {opacity: 0;}
            to {opacity: 1;}
        }
        .close-btn {
            color: #aaa;
            float: right;
            font-size: 28px;
            font-weight: bold;
            text-decoration: none;
            cursor: pointer;
        }
        .close-btn:hover,
        .close-btn:focus {
            color: #000;
            text-decoration: none;
            cursor: pointer;
        }
        /* ভাউচার নং-কে ক্লিকযোগ্য করার জন্য */
        .voucher-link {
            color: #007bff;
            font-weight: bold;
            cursor: pointer;
            text-decoration: underline;
            transition: color 0.2s;
        }
        .voucher-link:hover {
            color: #0056b3;
        }
        .loading-text {
            text-align: center;
            padding: 20px;
            color: #f39c12;
        }
        /* পপ-আপের ভিতরের টেবিল স্টাইল */
        .modal-details-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
            font-size: 0.9em; /* ছোট ফন্ট */
        }
        .modal-details-table th, .modal-details-table td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        .modal-details-table th {
            background-color: #e0f7fa;
            color: #34495e;
            text-align: center;
        }
        .modal-details-table td:nth-child(4) { /* Amount column */
             text-align: right;
             font-weight: bold;
        }
        .modal-details-table tfoot th {
             background-color: #ff9800;
             color: white;
             font-size: 1.1em;
             text-align: right;
        }
         /* Modal Footer স্টাইল */
.modal-footer {
    padding: 15px;
    border-top: 1px solid #dee2e6;
    display: flex;
    justify-content: flex-end;
    background-color: #f8f9fa;
}

.btn {
    padding: 10px 15px;
    border-radius: 4px;
    cursor: pointer;
    font-size: 1em;
}

.btn-primary {
    color: white;
    background-color: #007bff;
    border: 1px solid #007bff;
}

.btn-secondary {
    color: #333;
    background-color: #e9ecef;
    border: 1px solid #ccc;
}
        /* --- নতুন পপ-আপ (Modal) স্টাইল শেষ --- */

        /* প্রিন্ট মিডিয়া কোয়েরি (অপরিবর্তিত) */
        @media print {
            body { padding-top: 0 !important; background-color: white !important; }
            /* অপ্রয়োজনীয় অংশগুলো লুকানো */
            .header, .control-group, .search-form, a.back-button, .modal { 
                display: none !important; 
            }
            .report-container { box-shadow: none !important; margin: 0; padding: 0; }
            
            /* স্কুলের হেডার প্রিন্ট করার সময় দেখানো */
            .school-header-print { display: block !important; }
            
            .report-table { font-size: 10px; }
            .report-table th, .report-table td { padding: 5px 3px; border-color: #000; }
            .report-table th { 
                background-color: #ccc !important; color: #000 !important; 
                -webkit-print-color-adjust: exact; print-color-adjust: exact; 
            }
            .report-table .total-col, .report-table .grand-total-row th {
                background-color: #ff9800 !important;
                color: white !important;
                -webkit-print-color-adjust: exact; print-color-adjust: exact;
            }
            @page { 
                size: A4 landscape; 
                margin: 1cm; 
            }
        }
    </style>
</head>
<body>
<div class="header">
    <h1>ব্যয় রিপোর্ট (ভাউচার ভিত্তিক)</h1>
</div>

<a href="../dashboard.php" class="back-button">
    <i class="fas fa-arrow-left"></i> ড্যাশবোর্ড
</a>

<div class="report-container">
    
    <div class="school-header-print">
        <h3>মিফতাহুন নাজাত মহিলা মাদরাসা</h3>
        <p>নেয়ামতপুর,সদর,নোয়াখালী</p>
        <h4>ভাউচার ভিত্তিক ব্যয় রিপোর্ট</h4>
        <?php if (!empty($start_date) && !empty($end_date)): ?>
        <p style="font-weight: bold;">(তারিখ: <?php echo format_date_to_bangla($start_date); ?> থেকে <?php echo format_date_to_bangla($end_date); ?>)</p>
        <?php endif; ?>
    </div>

    <h2>ভাউচার নং এবং খাত অনুযায়ী ব্যয়ের সারাংশ</h2>

    <form method="GET" action="" class="search-form">
        <label for="start_date">শুরুর তারিখ:</label>
        <input type="date" id="start_date" name="start_date" required value="<?php echo htmlspecialchars($start_date); ?>">
        
        <label for="end_date">শেষের তারিখ:</label>
        <input type="date" id="end_date" name="end_date" required value="<?php echo htmlspecialchars($end_date); ?>">
        
        <button type="submit"><i class="fas fa-search"></i> সার্চ করুন</button>
        <?php if (!empty($start_date) || !empty($end_date)): ?>
            <button type="button" onclick="window.location.href='expense_report.php'" style="background-color: #dc3545;"><i class="fas fa-redo"></i> রিসেট</button>
        <?php endif; ?>
    </form>
    
    <div class="control-group">
        <a href="add_expense.php" class="expense-button"><i class="fas fa-plus-circle"></i> নতুন ব্যয় এন্ট্রি করুন</a>
        <button onclick="window.print()" class="print-button"><i class="fas fa-print"></i> প্রিন্ট করুন</button>
    </div>
    
    <?php if (empty($categories)): ?>
        <p style="text-align: center; margin-top: 30px; color: red; font-weight: bold;">⚠️ কোনো সক্রিয় ব্যয়ের খাত পাওয়া যায়নি। অনুগ্রহ করে 'ব্যয়ের খাত পরিচালনা' থেকে খাত যোগ করুন।</p>
    <?php elseif (empty($voucher_expenses) && (!empty($start_date) || !empty($end_date))): ?>
        <p style="text-align: center; margin-top: 30px; color: #f39c12; font-weight: bold;">নির্দিষ্ট তারিখ পরিসরে কোনো ব্যয় রেকর্ড পাওয়া যায়নি।</p>
    <?php elseif (empty($voucher_expenses)): ?>
        <p style="text-align: center; margin-top: 30px; color: #7f8c8d; font-weight: bold;">কোনো ব্যয় এন্ট্রি করা হয়নি।</p>
    <?php else: ?>
        <div style="overflow-x: auto;">
            <table class="report-table">
                <thead>
                    <tr>
                        <th class="fixed-col">ভাউচার নং</th>
                        <th class="fixed-col">তারিখ</th>
                        <th class="desc-col">বিবরণ</th>
                        <?php 
                        // ডাইনামিক কলাম হেডার: সব ব্যয়ের খাত
                        foreach ($categories as $cat_id => $cat_name): 
                        ?>
                            <th style="min-width: 80px;"><?php echo htmlspecialchars($cat_name); ?></th>
                        <?php endforeach; ?>
                        <th class="total-col">মোট ভাউচার ব্যয় (৳)</th>
                        <th style="min-width: 120px;">অ্যাকশন</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    foreach ($voucher_expenses as $voucher => $data):
                    ?>
                        <tr>
                            <td class="fixed-col">
                                <span class="voucher-link" data-voucher-no="<?php echo htmlspecialchars($voucher); ?>">
                                    <?php echo htmlspecialchars($voucher); ?>
                                </span>
                            </td>
                            <td class="fixed-col"><?php echo format_date_to_bangla($data['date']); ?></td>
                            <td class="desc-col"><?php echo htmlspecialchars($data['description']); ?></td>
                            <?php 
                            // প্রতি খাতের জন্য ভাউচার ভিত্তিক পরিমাণ
                            foreach ($category_ids as $cat_id):
                                $expense_amount = $data['categories'][$cat_id];
                            ?>
                                <td>
                                    <?php if ($expense_amount > 0): ?>
                                        <?php echo format_to_bangla_number($expense_amount); ?>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                            <?php endforeach; ?>
                            <td class="total-col">
                                <?php echo format_to_bangla_number($data['voucher_total']); ?>
                            </td>
                            
                            <td style="text-align: center;">
                                <a href="edit_expense.php?voucher_no=<?php echo urlencode($voucher); ?>" 
                                   title="সম্পাদনা করুন" class="action-link edit-btn">
                                    <i class="fas fa-edit"></i> এডিট
                                </a>
                                <a href="delete_expense.php?voucher_no=<?php echo urlencode($voucher); ?>" 
                                   onclick="return confirm('আপনি কি নিশ্চিত যে এই ভাউচার (<?php echo htmlspecialchars($voucher); ?>)-এর সব এন্ট্রি মুছে দিতে চান?');" 
                                   title="মুছে দিন" class="action-link delete-btn">
                                    <i class="fas fa-trash-alt"></i> ডিলেট
                                </a>
                            </td>
                        </tr>
                    <?php 
                    endforeach; 
                    ?>
                </tbody>
                <tfoot>
                    <tr class="grand-total-row">
                        <th colspan="3" style="text-align: right;">মোট গ্র্যান্ড টোটাল (খাত অনুযায়ী)</th>
                        <?php 
                        // কলামের গ্র্যান্ড টোটাল
                        foreach ($category_ids as $cat_id): 
                        ?>
                            <th><?php echo format_to_bangla_number($column_totals[$cat_id]); ?></th>
                        <?php endforeach; ?>
                        <th class="total-col" style="background-color: #ff9800; color: white;">
                            <?php echo format_to_bangla_number($grand_total_expenses); ?>
                        </th>
                        <th></th>
                    </tr>
                </tfoot>
            </table>
        </div>
    <?php endif; ?>

</div>

<div id="voucherModal" class="modal">
    <div class="modal-content">
        <span class="close-btn">&times;</span>
        <h3 id="modalVoucherTitle" style="text-align: center; color: #34495e; border-bottom: 2px solid #ddd; padding-bottom: 10px; margin-bottom: 15px;"></h3>
        <div id="modalVoucherBody">
            <div class="loading-text"><i class="fas fa-spinner fa-spin"></i> লোড হচ্ছে...</div>
            </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const modal = document.getElementById('voucherModal');
    const closeBtn = document.querySelector('.close-btn');
    const links = document.querySelectorAll('.voucher-link'); 
    const modalBody = document.getElementById('modalVoucherBody');
    const modalTitle = document.getElementById('modalVoucherTitle');
    
    // modal-footer বাটন রাখার জন্য একটি নতুন বিভাগ তৈরি করুন
    const modalFooter = document.createElement('div');
    modalFooter.className = 'modal-footer'; // CSS-এর জন্য একটি ক্লাস
    modalFooter.innerHTML = `
        <button id="printVoucherBtn" class="btn btn-primary" style="background-color: #28a745; border-color: #28a745; margin-right: 10px;">
            <i class="fas fa-print"></i> প্রিন্ট করুন
        </button>
        <button type="button" class="btn btn-secondary close-btn">বন্ধ করুন</button>
    `;

    // Modal-এ Footer যোগ করা
    // ধরে নিচ্ছি modal-content-এর ভেতরেই modalBody আছে
    if (modal) {
        const modalContent = modal.querySelector('.modal-content');
        if (modalContent) {
            modalContent.appendChild(modalFooter);
        }
    }

    // পপ-আপ বন্ধ করার ইভেন্ট
    modalFooter.querySelector('.close-btn').onclick = function() {
        modal.style.display = 'none';
    }
    
    // উইন্ডোর বাইরে ক্লিক করলে পপ-আপ বন্ধ হবে
    window.onclick = function(event) {
        if (event.target === modal) {
            modal.style.display = 'none';
        }
    }

    // === প্রিন্ট ফাংশন ===
    // এটি শুধুমাত্র voucher-container-কে প্রিন্ট করবে
    document.getElementById('printVoucherBtn').addEventListener('click', function() {
        // modalBody-এর মধ্যে থাকা voucher-container-এর HTML নিন
        const printContent = modalBody.innerHTML;
        
        // একটি নতুন উইন্ডো তৈরি করুন
        const printWindow = window.open('', '', 'height=600,width=800');
        
        // প্রিন্ট উইন্ডোর হেড অংশে স্টাইল যুক্ত করুন (প্রয়োজনে আপনার মূল CSS লিংকগুলোও যুক্ত করতে পারেন)
        printWindow.document.write('<html><head><title>ব্যয় ভাউচার প্রিন্ট</title>');
        
        // আপনার CSS স্টাইলগুলো এখানে কপি করে দিন
        printWindow.document.write(`
            <style>
                body { font-family: Arial, 'Kalpurush', sans-serif; margin: 0; padding: 20px; }
                .voucher-container { padding: 0; background-color: #ffffff; color: #333; border: 1px solid #ccc; }
                .voucher-header { text-align: center; border-bottom: 3px solid #007bff; padding-bottom: 10px; margin-bottom: 20px; }
                .voucher-title { color: #007bff; margin: 0; font-size: 1.8em; font-weight: bold; }
                .voucher-info p { margin: 5px 0; font-size: 1em; }
                
                /* Table Style */
                .modal-details-table { width: 100%; border-collapse: collapse; margin-top: 15px; font-size: 0.95em; }
                .modal-details-table thead th { background-color: #343a40 !important; color: white; padding: 10px; text-align: left; border: 1px solid #343a40; }
                .modal-details-table tbody td { padding: 8px 10px; border: 1px solid #e9ecef; vertical-align: top; }
                .modal-details-table tbody tr:nth-child(even) { background-color: #f8f9fa; }
                
                /* Total Row Style */
                .modal-details-table tfoot th { 
                    background-color: #e67e22 !important; 
                    color: white; 
                    padding: 12px 10px; 
                    text-align: right; 
                    font-size: 1.2em;
                }
                
                /* Specific style for Teacher/Salary details within 'বিবরণ' column */
                .teacher-detail-box {
                    margin: 5px 0;
                    padding: 5px;
                    border-left: 3px solid #28a745;
                    background-color: #e2f0e6;
                    border-radius: 3px;
                }
                .teacher-detail-box .name { font-weight: bold; color: #1a7431; display: block; }
                .teacher-detail-box .month { font-size: 0.9em; color: #383d41; display: block; }
                .voucher-footer { border-top: 1px dashed #ccc; padding-top: 15px; margin-top: 20px; text-align: right; font-size: 0.9em; color: #6c757d; }
                
                /* প্রিন্টের জন্য অপ্রয়োজনীয় অংশ লুকানো */
                @media print {
                    /* যদি কোন unnecessary element থাকে */
                }

            </style>
        `);
        
        printWindow.document.write('</head><body>');
        printWindow.document.write(printContent); // শুধুমাত্র ভাউচার কন্টেন্ট যোগ করা
        printWindow.document.write('</body></html>');
        printWindow.document.close();
        
        // প্রিন্ট ডায়ালগ ওপেন করা
        printWindow.print();
        // প্রয়োজন হলে প্রিন্টের পর উইন্ডো বন্ধ করা
        // printWindow.close(); 
    });

    // ভাউচার নং-এর ওপর ক্লিক করলে ইভেন্ট হ্যান্ডলার (আগের লজিক)
    links.forEach(link => {
        link.addEventListener('click', function() {
            // ... (বাকি AJAX লোডিং লজিক অপরিবর্তিত) ...
            const voucherNo = this.getAttribute('data-voucher-no');
            modalTitle.textContent = `ভাউচার নং: ${voucherNo} এর বিস্তারিত`;
            modalBody.innerHTML = '<div class="loading-text" style="text-align: center; padding: 20px;"><i class="fas fa-spinner fa-spin"></i> বিস্তারিত লোড হচ্ছে...</div>';
            modal.style.display = 'block';
            
            // AJAX কল করে ভাউচার বিস্তারিত লোড করা
            fetch('fetch_voucher_details.php?voucher_no=' + encodeURIComponent(voucherNo))
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        modalBody.innerHTML = `<p style="color: red; text-align: center;">🛑 এরর: ${data.error}</p>`;
                        return;
                    }
                    
                    // === ভাউচার-সদৃশ HTML কাঠামো তৈরি করা শুরু ===
                    let html = `
                        <div class="voucher-container">
                            <div class="voucher-header">
                                <h4 class="voucher-title">ব্যয় ভাউচার</h4>
                            </div>
                            
                            <div class="voucher-info" style="display: flex; justify-content: space-between; flex-wrap: wrap; margin-bottom: 15px;">
                                <p><strong>ভাউচার নং:</strong> ${data.voucher_no}</p>
                                <p><strong>তারিখ:</strong> ${data.expense_date}</p>
                                ${data.paid_to ? `<p><strong>যাকে পরিশোধ:</strong> ${data.paid_to}</p>` : ''}
                            </div>
                            
                            ${data.main_description ? `<p style="margin-bottom: 15px;"><strong>মূল/একটি বিবরণ:</strong> ${data.main_description}</p>` : ''}

                            <h5 style="color: #343a40; border-bottom: 2px solid #ced4da; padding-bottom: 5px;">লেনদেনের বিস্তারিত:</h5>
                    `;
                    
                    // === সমস্ত খরচের বিস্তারিত অংশ ===
                    if (data.details && data.details.length > 0) {
                        html += `
                            <table class="modal-details-table">
                                <thead>
                                    <tr>
                                        <th style="width: 5%;">ক্রমিক</th>
                                        <th style="width: 25%;">খাত</th>
                                        <th style="width: 50%;">বিবরণ</th>
                                        <th style="width: 20%; text-align: right;">পরিমাণ (৳)</th>
                                    </tr>
                                </thead>
                                <tbody>
                        `;
                        
                        data.details.forEach(detail => {
                            const entryDescription = detail.entry_description;
                            html += `
                                <tr>
                                    <td style="text-align: center;">${detail.serial}</td>
                                    <td>${detail.category_name}</td>
                                    <td>${entryDescription}</td> 
                                    <td style="text-align: right; font-weight: bold;">${detail.amount}</td>
                                </tr>
                            `;
                        });

                        html += `
                                </tbody>
                                <tfoot>
                                    <tr>
                                        <th colspan="3">মোট ভাউচার পরিশোধিত</th>
                                        <th style="text-align: right;">${data.total_amount}</th>
                                    </tr>
                                </tfoot>
                            </table>
                        `;
                    } else {
                        html += '<p style="text-align: center; color: #dc3545; padding: 15px;">এই ভাউচার নং এর জন্য কোনো বিস্তারিত এন্ট্রি পাওয়া যায়নি।</p>';
                    }
                    
                    // === ফুটার (এন্ট্রি করেছেন) ===
                    html += `
                            <div class="voucher-footer">
                                <strong>এন্ট্রি করেছেন:</strong> ${data.received_by_name}
                            </div>
                        </div>
                    `;

                    modalBody.innerHTML = html;
                })
                .catch(error => {
                    console.error('Fetch error:', error);
                    modalBody.innerHTML = `<p style="color: red; text-align: center;">🛑 ডেটা লোড করার সময় একটি সমস্যা হয়েছে।</p>`;
                });
        });
    });
});
</script>

</body>
</html>