<?php
session_start();
include '../../includes/db_connection.php'; 

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'Admin') {
    die("আপনার এই পাতা ব্যবহারের অনুমতি নেই।");
}

$voucher_no = $_GET['voucher_no'] ?? '';
if (empty($voucher_no)) {
    header("Location: expense_report.php");
    exit();
}

$message = '';
$error = '';
$current_expenses = [];
$expense_categories = [];
$initial_date = date('Y-m-d');
$initial_description = '';

// ১. ডাটাবেস থেকে সক্রিয় ব্যয়ের খাতের তালিকা লোড করা
$categories_sql = "SELECT category_id, category_name FROM expense_categories WHERE is_active = TRUE ORDER BY category_name ASC";
$categories_result = $conn->query($categories_sql);
if ($categories_result && $categories_result->num_rows > 0) {
    while ($row = $categories_result->fetch_assoc()) {
        $expense_categories[$row['category_id']] = $row['category_name'];
    }
}

// ২. বর্তমান ভাউচারের ডেটা লোড করা
$load_sql = "SELECT expense_id, category_id, expense_date, description, amount FROM expenses WHERE voucher_no = ?";
$stmt_load = $conn->prepare($load_sql);
$stmt_load->bind_param("s", $voucher_no);
$stmt_load->execute();
$result_load = $stmt_load->get_result();

if ($result_load->num_rows > 0) {
    while ($row = $result_load->fetch_assoc()) {
        // একটি ভাউচারে একাধিক এন্ট্রি থাকতে পারে, তাই array-তে লোড করা হলো
        $current_expenses[$row['expense_id']] = $row;
        $initial_date = $row['expense_date'];
        $initial_description = $row['description']; 
        // বিবরণ সব এন্ট্রির জন্য একই ধরছি
    }
} else {
    $error = "এই ভাউচার নম্বরের (" . htmlspecialchars($voucher_no) . ") জন্য কোনো এন্ট্রি পাওয়া যায়নি।";
}
$stmt_load->close();


// ৩. ফর্ম সাবমিট হ্যান্ডেল করা (আপডেট লজিক)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_expense'])) {
    $new_date = $_POST['expense_date'] ?? $initial_date;
    $new_description = trim($_POST['description'] ?? $initial_description);

    $conn->begin_transaction();
    $update_success = true;

    // A. ভাউচারের সাধারণ তথ্য (তারিখ, বিবরণ) আপডেট করা
    $update_voucher_details_sql = "UPDATE expenses SET expense_date = ?, description = ? WHERE voucher_no = ?";
    $stmt_detail = $conn->prepare($update_voucher_details_sql);
    $stmt_detail->bind_param("sss", $new_date, $new_description, $voucher_no);
    if (!$stmt_detail->execute()) {
        $update_success = false;
        $error = "সাধারণ তথ্য আপডেটে ব্যর্থ: " . $stmt_detail->error;
    }
    $stmt_detail->close();
    
    // B. প্রতিটি এন্ট্রি লুপ করে পরিমাণ আপডেট করা
    if ($update_success) {
        foreach ($current_expenses as $expense_id => $data) {
            $new_amount = $_POST['amount'][$expense_id] ?? 0;
            $new_amount = (float)$new_amount;

            $update_sql = "UPDATE expenses SET amount = ? WHERE expense_id = ?";
            $stmt_update = $conn->prepare($update_sql);
            $stmt_update->bind_param("di", $new_amount, $expense_id);
            
            if (!$stmt_update->execute()) {
                $update_success = false;
                $error = "ব্যয় আইডি: " . $expense_id . " আপডেটে ব্যর্থ: " . $stmt_update->error;
                break;
            }
            $stmt_update->close();
        }
    }

    if ($update_success) {
        $conn->commit();
        $message = "ভাউচার নং " . htmlspecialchars($voucher_no) . " সফলভাবে আপডেট করা হয়েছে!";
        // আপডেট হওয়ার পর পেজ রিফ্রেশ করে নতুন ডেটা লোড করা
        header("Location: edit_expense.php?voucher_no=" . urlencode($voucher_no) . "&status=success");
        exit();
    } else {
        $conn->rollback();
        // যদি কোনো ত্রুটি হয়, error ভ্যারিয়েবল ব্যবহার করা হয়েছে
    }
}

// ৪. স্ট্যাটাস চেক (রিফ্রেশের পরে মেসেজ দেখানোর জন্য)
if (isset($_GET['status']) && $_GET['status'] == 'success') {
    $message = "ভাউচার নং " . htmlspecialchars($voucher_no) . " সফলভাবে আপডেট করা হয়েছে!";
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ব্যয় সম্পাদনা (ভাউচার: <?php echo htmlspecialchars($voucher_no); ?>)</title>
    <link rel="stylesheet" href="../../public/css/style.css"> 
    <style>
        .form-container { max-width: 600px; margin: 50px auto; padding: 20px; border: 1px solid #ccc; border-radius: 5px; background-color: #f9f9f9; }
        .form-container label { display: block; margin-bottom: 5px; font-weight: bold; }
        .form-container input[type="text"], .form-container input[type="number"], .form-container input[type="date"], .form-container textarea { 
            width: 100%; padding: 10px; margin-bottom: 15px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; 
        }
        .category-item { display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; padding: 10px; border: 1px solid #e0e0e0; border-radius: 4px; background-color: #fff; }
        .category-item label { flex: 2; margin-bottom: 0; }
        .category-item input { flex: 1; width: auto; margin-left: 10px; text-align: right; }
        .message-success { color: green; font-weight: bold; margin-bottom: 15px; }
        .message-error { color: red; font-weight: bold; margin-bottom: 15px; }
    </style>
</head>
<body>
<div class="header">
    <h1>ব্যয় সম্পাদনা</h1>
</div>
<div class="content">
    <div class="form-container">
        <h2>ভাউচার নং: <?php echo htmlspecialchars($voucher_no); ?></h2>
        <a href="expense_report.php" style="float: right; text-decoration: none; color: #007bff;">← রিপোর্ট</a>
        <div style="clear: both;"></div>

        <?php if ($message): ?>
            <p class="message-success"><?php echo $message; ?></p>
        <?php endif; ?>
        <?php if ($error): ?>
            <p class="message-error"><?php echo $error; ?></p>
        <?php endif; ?>

        <?php if (!empty($current_expenses)): ?>
        <form action="" method="POST">
            <label for="expense_date">ব্যয়ের তারিখ:</label>
            <input type="date" id="expense_date" name="expense_date" required value="<?php echo htmlspecialchars($initial_date); ?>">
            
            <label for="description">বিস্তারিত বিবরণ:</label>
            <textarea id="description" name="description" rows="3"><?php echo htmlspecialchars($initial_description); ?></textarea>

            <h3 style="margin-top: 20px;">খাত অনুযায়ী পরিমাণ (পরিবর্তন করুন)</h3>
            
            <?php foreach ($current_expenses as $expense_id => $data): ?>
                <div class="category-item">
                    <label>
                        <?php echo htmlspecialchars($expense_categories[$data['category_id']] ?? 'Unknown Category'); ?>
                    </label>
                    <input type="number" step="0.01" name="amount[<?php echo $expense_id; ?>]" 
                           value="<?php echo htmlspecialchars($data['amount']); ?>" required min="0">
                </div>
            <?php endforeach; ?>

            <button type="submit" name="update_expense" style="background-color: #3498db; color: white; padding: 10px 15px; border: none; border-radius: 4px; cursor: pointer; margin-top: 20px;">
                পরিবর্তন সেভ করুন
            </button>
        </form>
        <?php endif; ?>
    </div>
</div>
</body>
</html>