<?php
session_start();
include '../../includes/db_connection.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: /school_management/public/index.php");
    exit();
}

// Fixed exam ID for "দ্বিতীয় সাময়িক পরীক্ষা" (adjust if your ID is different)
$exam_id = 2;
$exam_name = "দ্বিতীয় সাময়ীক পরিক্ষা";

$students = [];
$subjects = [];
$exam_results = [];
$total_marks_by_student = [];
$subject_count_by_student = []; 
$students_with_results = []; 
$class_id = $_GET['class_id'] ?? '';
$class_name = '';

if ($class_id) {
    // ----------------------------------------------------------------
    // A. শিক্ষার্থী, মোট নম্বর ও বিষয় সংখ্যা গণনা
    // ----------------------------------------------------------------
    $sql_students = "SELECT s.student_id, s.first_name, s.last_name, s.class_roll, c.class_name 
                     FROM students s
                     LEFT JOIN classes c ON s.class_id = c.class_id
                     WHERE s.class_id = ? 
                     ORDER BY s.class_roll";
    $stmt_students = $conn->prepare($sql_students);

    if ($stmt_students === false) {
        die("Error preparing student SQL: " . $conn->error);
    }

    $stmt_students->bind_param("i", $class_id);
    $stmt_students->execute();
    $students_result = $stmt_students->get_result();

    if ($students_result->num_rows > 0) {
        $first_student = $students_result->fetch_assoc();
        $class_name = $first_student['class_name'];
        $students_result->data_seek(0); 
        while ($row = $students_result->fetch_assoc()) {
            $id = $row['student_id'];
            $students[$id] = $row;
            // মোট নম্বর এবং বিষয় সংখ্যা গণনা শুরু
            $total_marks_by_student[$id] = 0;
            $subject_count_by_student[$id] = 0; 
        }
    }
    $stmt_students->close();

    // সকল বিষয় লোড করা
    $subjects_sql = "SELECT subject_name FROM subjects ORDER BY subject_id";
    $subjects_result = $conn->query($subjects_sql);
    while ($row = $subjects_result->fetch_assoc()) {
        $subjects[] = $row['subject_name'];
    }

    // পরীক্ষার ফলাফল লোড করা
    $sql_results = "SELECT er.student_id, sub.subject_name, er.marks 
                     FROM exam_results er
                     LEFT JOIN subjects sub ON er.subject_id = sub.subject_id
                     WHERE er.exam_id = ? AND er.student_id IN (SELECT student_id FROM students WHERE class_id = ?)";
    $stmt_results = $conn->prepare($sql_results);
    
    if ($stmt_results === false) {
        die("Error preparing result SQL: " . $conn->error);
    }

    $stmt_results->bind_param("ii", $exam_id, $class_id);
    $stmt_results->execute();
    $results_result = $stmt_results->get_result();

    // মোট নম্বর এবং বিষয় সংখ্যা গণনা
    while ($row = $results_result->fetch_assoc()) {
        $student_id = $row['student_id'];
        $subject_name = $row['subject_name'];
        $marks = (int)$row['marks']; 

        $exam_results[$student_id][$subject_name] = $marks;
        
        if (isset($total_marks_by_student[$student_id])) {
            $total_marks_by_student[$student_id] += $marks;
            $subject_count_by_student[$student_id]++; 
        }
    }
    $stmt_results->close();

    // সকল শিক্ষার্থীর তথ্য, ফলাফল এবং মোট নম্বর একত্রিত করা ও গড় হিসাব
    foreach ($students as $student_id => $data) {
        $total_marks = $total_marks_by_student[$student_id];
        $subject_count = $subject_count_by_student[$student_id];
        
        $average_marks = ($subject_count > 0) ? number_format($total_marks / $subject_count, 2) : 'N/A';
        
        $students_with_results[] = [
            'student_id' => $student_id,
            'class_roll' => $data['class_roll'],
            'student_name' => $data['first_name'] . ' ' . $data['last_name'],
            'total_marks' => $total_marks,
            'average_marks' => $average_marks,
            'exam_marks' => $exam_results[$student_id] ?? [],
            'position' => 0 
        ];
    }

    // ----------------------------------------------------------------
    // B. অবস্থান নির্ধারণ (Ranking Logic)
    // ----------------------------------------------------------------

    // ১. মোট নম্বরের ভিত্তিতে উচ্চক্রম থেকে নিম্নক্রমে সাজানো (Descending)
    usort($students_with_results, function($a, $b) {
        if ($a['total_marks'] == 0 && $b['total_marks'] != 0) return 1;
        if ($a['total_marks'] != 0 && $b['total_marks'] == 0) return -1;
        if ($a['total_marks'] == 0 && $b['total_marks'] == 0) return 0;
        
        if ($a['total_marks'] == $b['total_marks']) {
            return $a['class_roll'] <=> $b['class_roll'];
        }
        return $b['total_marks'] <=> $a['total_marks']; 
    });

    // ২. সাজানোর পর অবস্থান (Rank) নির্ধারণ করা (Tied Rank)
    $rank = 0;
    $prev_marks = -1;
    $tie_counter = 1;

    foreach ($students_with_results as $key => &$student) {
        if ($student['total_marks'] > 0) {
            if ($student['total_marks'] != $prev_marks) {
                $rank += $tie_counter;
                $tie_counter = 1;
            } else {
                $tie_counter++;
            }
            $student['position'] = $rank;
            $prev_marks = $student['total_marks'];
        } else {
            $student['position'] = 'N/A';
        }
    }
    unset($student); 
}

// ----------------------------------------------------------------
// C. সকল ক্লাস লোড করা (ড্রপডাউনের জন্য)
// ----------------------------------------------------------------
$classes_sql = "SELECT class_id, class_name FROM classes ORDER BY class_name";
$classes_result = $conn->query($classes_sql);
?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($exam_name); ?> ফলাফল</title>
    <link rel="stylesheet" href="/school_management/public/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    
    <style>
        /* ফন্ট স্টাইল */
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            background-color: #f4f7f6;
        }
        .header {
            background-color: #34495e;
            color: white;
            padding: 10px 0;
            text-align: center;
        }
        .nav a {
            color: white;
            margin: 0 10px;
            text-decoration: none;
        }
        
        /* কন্টেইনার ফিক্স - পেজের বাইরে ডেটা যাওয়া বন্ধ করার জন্য */
        .container {
            width: 98%; 
            max-width: 1400px; 
            margin: 0 auto;
            padding: 20px;
            background-color: white;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
            overflow-x: auto; /* টেবিল স্ক্রলের জন্য */
        }

        .report-header { text-align: center; margin-bottom: 20px; }
        .report-header h1 { font-size: 2.5em; margin-bottom: 5px; }
        .report-header h2 { font-size: 1.5em; margin: 5px 0; }
        .report-header h3 { font-size: 1.2em; margin: 5px 0; font-weight: normal; }

        .report-table { 
            width: 100%; 
            border-collapse: collapse; 
            margin-top: 20px; 
            min-width: 1000px; /* কলাম বেশি হলে টেবিলের ন্যূনতম চওড়া */
        }
        .report-table th, .report-table td { 
            border: 1px solid #000; 
            padding: 8px; 
            text-align: center; 
            white-space: nowrap; 
        }
        .report-table th { background-color: #f2f2f2; font-weight: bold; }
        
        /* কলামের স্টাইল */
        .rank-col { background-color: #fff3e0; font-weight: bold; color: #ff9800; } 
        .total-col { background-color: #e8f5e9; font-weight: bold; color: #2e7d32; } 
        .average-col { background-color: #e3f2fd; font-weight: bold; color: #1e88e5; } 
        .report-table td:nth-child(3) { 
            text-align: left;
            min-width: 150px; 
        }
        
        .filter-form { text-align: center; margin-bottom: 20px; }
        .filter-form select {
            padding: 8px 12px;
            border-radius: 4px;
            border: 1px solid #ccc;
            margin-right: 10px;
        }
        .print-btn {
            background-color: #3498db;
            color: white;
            border: none;
            padding: 8px 15px;
            border-radius: 4px;
            cursor: pointer;
            font-weight: bold;
            margin-bottom: 15px;
        }

        /* ----------------------------------------------------------------- */
        /* **ল্যান্ডস্কেপ প্রিন্ট স্টাইল** */
        /* ----------------------------------------------------------------- */
        @media print {
            @page {
                size: A4 landscape; /* A4 পেপারে ল্যান্ডস্কেপ মোড সেট করা */
                margin: 10mm; 
            }
            
            .header, .nav, .footer, .filter-form, .print-btn { 
                display: none; 
            }
            .container { 
                width: 100%; 
                margin: 0; 
                padding: 0; 
                overflow-x: hidden;
            }
            .report-header { 
                padding-top: 10px; 
            }
            .report-table { 
                font-size: 9pt; 
                min-width: 100%; 
            }
        }
    </style>
</head>
<body>
<div class="header">
    <div class="nav">
        <a href="/school_management/public/dashboard.php">Dashboard</a>
        <a href="/school_management/public/students/students_list.php">Students</a>
        <a href="/school_management/public/teachers/teachers_list.php">Teachers</a>
        <a href="/school_management/public/fees/fee_collection.php">Fees</a>
        <a href="/school_management/public/fees/monthly_fee_summary.php">Monthly Report</a>
        <a href="/school_management/public/exam_results/add_exam_result.php">Exams</a>
        <a href="/school_management/public/exam_results/exam_report.php">Exam Report</a>
        <a href="/school_management/public/attendance/record_attendance.php">Attendance</a>
        <a href="/school_management/public/attendance/attendance_report.php">Attendance Report</a>
        <a href="/school_management/public/change_password.php">Change Password</a>
        <a href="/school_management/public/logout.php">Logout</a>
    </div>
</div>

<div class="content">
    <div class="container">
        <div class="filter-form">
            <form action="" method="GET">
                <label for="class_id">শ্রেণি নির্বাচন করুন:</label>
                <select id="class_id" name="class_id" onchange="this.form.submit()" required>
                    <option value="">--একটি শ্রেণি নির্বাচন করুন--</option>
                    <?php
                    if ($classes_result && $classes_result->num_rows > 0) {
                        $classes_result->data_seek(0);
                        while($row = $classes_result->fetch_assoc()) {
                            $selected = ($class_id == $row['class_id']) ? 'selected' : '';
                            echo "<option value='" . htmlspecialchars($row['class_id']) . "' " . $selected . ">" . htmlspecialchars($row['class_name']) . "</option>";
                        }
                    }
                    ?>
                </select>
            </form>
        </div>

        <?php if ($class_id && !empty($students_with_results)): ?>
            <div class="report-header">
                <h1>মিফতাহুন নাজাত মহিলা মাদরাসা</h1>
                <p>নেয়ামতপুর সদর, নোয়াখালী</p>
                <h2><?php echo htmlspecialchars($exam_name); ?></h2>
                <h3>শ্রেণী: <?php echo htmlspecialchars($class_name); ?></h3>
            </div>
            <button onclick="window.print()" class="print-btn"><i class="fas fa-print"></i> রিপোর্ট প্রিন্ট করুন</button>
            <table class="report-table">
                <thead>
                    <tr>
                        <th class="rank-col">অবস্থান</th> 
                        <th>ক্লাস রোল</th>
                        <th>শিক্ষার্থীর নাম</th>
                        <?php foreach ($subjects as $subject): ?>
                            <th><?php echo htmlspecialchars($subject); ?></th>
                        <?php endforeach; ?>
                        <th class="total-col">মোট নম্বর</th> 
                        <th class="average-col">গড় নম্বর</th> 
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($students_with_results as $student): ?>
                        <tr>
                            <td class="rank-col">
                                <?php echo htmlspecialchars($student['position']); ?>
                            </td>
                            <td><?php echo htmlspecialchars($student['class_roll']); ?></td>
                            <td style="text-align: left;"><?php echo htmlspecialchars($student['student_name']); ?></td>
                            <?php 
                            foreach ($subjects as $subject): 
                                $marks = $student['exam_marks'][$subject] ?? '-';
                            ?>
                                <td>
                                    <?php echo htmlspecialchars($marks); ?>
                                </td>
                            <?php endforeach; ?>
                            
                            <td class="total-col">
                                <?php echo htmlspecialchars($student['total_marks'] ?? 0); ?>
                            </td>
                            
                            <td class="average-col">
                                <?php echo htmlspecialchars($student['average_marks']); ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php elseif ($class_id): ?>
            <p style="text-align: center;">এই শ্রেণীতে কোনো শিক্ষার্থী বা পরীক্ষার ফলাফল পাওয়া যায়নি।</p>
        <?php else: ?>
             <p style="text-align: center; margin-top: 30px; font-weight: bold;">উপরে ড্রপডাউন থেকে একটি শ্রেণি নির্বাচন করুন ফলাফল দেখার জন্য।</p>
        <?php endif; ?>
    </div>
</div>

<div class="footer">
    <p>&copy; <?php echo date("Y"); ?> School Management System. All Rights Reserved.</p>
</div>
</body>
</html>
<?php
// ডাটাবেস সংযোগ বন্ধ করা
if (isset($conn)) {
    $conn->close();
}
?>